/*
** Copyright (c) 2025 LunarG, Inc.
**
** Permission is hereby granted, free of charge, to any person obtaining a
** copy of this software and associated documentation files (the "Software"),
** to deal in the Software without restriction, including without limitation
** the rights to use, copy, modify, merge, publish, distribute, sublicense,
** and/or sell copies of the Software, and to permit persons to whom the
** Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in
** all copies or substantial portions of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
** AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
** LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
** FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
*/

#include <catch2/catch.hpp>

#include "util/logging.h"
#include "util/spirv_parsing_util.h"

#if 0 // original glsl: https://github.com/SaschaWillems/Vulkan/blob/master/shaders/glsl/raytracinggltf/closesthit.rchit
/* Copyright (c) 2023, Sascha Willems
 *
 * SPDX-License-Identifier: MIT
 *
 */

#version 460

#extension GL_EXT_ray_tracing : require
#extension GL_GOOGLE_include_directive : require
#extension GL_EXT_nonuniform_qualifier : require
#extension GL_EXT_buffer_reference2 : require
#extension GL_EXT_scalar_block_layout : require
#extension GL_EXT_shader_explicit_arithmetic_types_int64 : require

layout(location = 0) rayPayloadInEXT vec3 hitValue;
layout(location = 2) rayPayloadEXT bool shadowed;
hitAttributeEXT vec2 attribs;

layout(binding = 0, set = 0) uniform accelerationStructureEXT topLevelAS;
layout(binding = 3, set = 0) uniform sampler2D image;

struct GeometryNode {
	uint64_t vertexBufferDeviceAddress;
	uint64_t indexBufferDeviceAddress;
	int textureIndexBaseColor;
	int textureIndexOcclusion;
};
layout(binding = 4, set = 0) buffer GeometryNodes { GeometryNode nodes[]; } geometryNodes;

layout(binding = 5, set = 0) uniform sampler2D textures[];

#include "bufferreferences.glsl"
#include "geometrytypes.glsl"

void main()
{
	Triangle tri = unpackTriangle(gl_PrimitiveID, 112);
	hitValue = vec3(tri.normal);

	GeometryNode geometryNode = geometryNodes.nodes[gl_GeometryIndexEXT];

	vec3 color = texture(textures[nonuniformEXT(geometryNode.textureIndexBaseColor)], tri.uv).rgb;
	if (geometryNode.textureIndexOcclusion > -1) {
		float occlusion = texture(textures[nonuniformEXT(geometryNode.textureIndexOcclusion)], tri.uv).r;
		color *= occlusion;
	}

	hitValue = color;

	// Shadow casting
	float tmin = 0.001;
	float tmax = 10000.0;
	float epsilon = 0.001;
	vec3 origin = gl_WorldRayOriginEXT + gl_WorldRayDirectionEXT * gl_HitTEXT + tri.normal * epsilon;
	shadowed = true;
	vec3 lightVector = vec3(-5.0, -2.5, -5.0);
	// Trace shadow ray and offset indices to match shadow hit/miss shader group indices
//	traceRayEXT(topLevelAS, gl_RayFlagsTerminateOnFirstHitEXT | gl_RayFlagsOpaqueEXT | gl_RayFlagsSkipClosestHitShaderEXT, 0xFF, 0, 0, 1, origin, tmin, lightVector, tmax, 2);
//	if (shadowed) {
//		hitValue *= 0.7;
//	}
}
#endif

const uint32_t spirv_bda_array[] = {
    0x07230203, 0x00010500, 0x0008000b, 0x000001da, 0x00000000, 0x00020011, 0x0000000b, 0x00020011, 0x0000117f,
    0x00020011, 0x000014b5, 0x00020011, 0x000014b6, 0x00020011, 0x000014bb, 0x00020011, 0x000014e3, 0x0006000a,
    0x5f565053, 0x5f52484b, 0x5f796172, 0x63617274, 0x00676e69, 0x0006000b, 0x00000001, 0x4c534c47, 0x6474732e,
    0x3035342e, 0x00000000, 0x0003000e, 0x000014e4, 0x00000001, 0x000c000f, 0x000014c4, 0x00000004, 0x6e69616d,
    0x00000000, 0x00000021, 0x00000024, 0x0000007c, 0x000000b2, 0x000000ba, 0x000000c7, 0x000000fe, 0x00030003,
    0x00000002, 0x000001cc, 0x00070004, 0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x0065636e,
    0x00080004, 0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x3265636e, 0x00000000, 0x00080004,
    0x455f4c47, 0x6e5f5458, 0x6e756e6f, 0x726f6669, 0x75715f6d, 0x66696c61, 0x00726569, 0x00060004, 0x455f4c47,
    0x725f5458, 0x745f7961, 0x69636172, 0x0000676e, 0x00080004, 0x455f4c47, 0x735f5458, 0x616c6163, 0x6c625f72,
    0x5f6b636f, 0x6f79616c, 0x00007475, 0x000d0004, 0x455f4c47, 0x735f5458, 0x65646168, 0x78655f72, 0x63696c70,
    0x615f7469, 0x68746972, 0x6974656d, 0x79745f63, 0x5f736570, 0x36746e69, 0x00000034, 0x000a0004, 0x475f4c47,
    0x4c474f4f, 0x70635f45, 0x74735f70, 0x5f656c79, 0x656e696c, 0x7269645f, 0x69746365, 0x00006576, 0x00080004,
    0x475f4c47, 0x4c474f4f, 0x6e695f45, 0x64756c63, 0x69645f65, 0x74636572, 0x00657669, 0x00040005, 0x00000004,
    0x6e69616d, 0x00000000, 0x00040005, 0x0000000d, 0x74726556, 0x00007865, 0x00040006, 0x0000000d, 0x00000000,
    0x00736f70, 0x00050006, 0x0000000d, 0x00000001, 0x6d726f6e, 0x00006c61, 0x00040006, 0x0000000d, 0x00000002,
    0x00007675, 0x00060005, 0x0000001a, 0x6d6f6547, 0x79727465, 0x65646f4e, 0x00000000, 0x000a0006, 0x0000001a,
    0x00000000, 0x74726576, 0x75427865, 0x72656666, 0x69766544, 0x64416563, 0x73657264, 0x00000073, 0x000a0006,
    0x0000001a, 0x00000001, 0x65646e69, 0x66754278, 0x44726566, 0x63697665, 0x64644165, 0x73736572, 0x00000000,
    0x00090006, 0x0000001a, 0x00000002, 0x74786574, 0x49657275, 0x7865646e, 0x65736142, 0x6f6c6f43, 0x00000072,
    0x00090006, 0x0000001a, 0x00000003, 0x74786574, 0x49657275, 0x7865646e, 0x6c63634f, 0x6f697375, 0x0000006e,
    0x00060005, 0x0000001d, 0x6d6f6547, 0x79727465, 0x65646f4e, 0x00000000, 0x000a0006, 0x0000001d, 0x00000000,
    0x74726576, 0x75427865, 0x72656666, 0x69766544, 0x64416563, 0x73657264, 0x00000073, 0x000a0006, 0x0000001d,
    0x00000001, 0x65646e69, 0x66754278, 0x44726566, 0x63697665, 0x64644165, 0x73736572, 0x00000000, 0x00090006,
    0x0000001d, 0x00000002, 0x74786574, 0x49657275, 0x7865646e, 0x65736142, 0x6f6c6f43, 0x00000072, 0x00090006,
    0x0000001d, 0x00000003, 0x74786574, 0x49657275, 0x7865646e, 0x6c63634f, 0x6f697375, 0x0000006e, 0x00060005,
    0x0000001f, 0x6d6f6547, 0x79727465, 0x65646f4e, 0x00000073, 0x00050006, 0x0000001f, 0x00000000, 0x65646f6e,
    0x00000073, 0x00060005, 0x00000021, 0x6d6f6567, 0x79727465, 0x65646f4e, 0x00000073, 0x00070005, 0x00000024,
    0x475f6c67, 0x656d6f65, 0x49797274, 0x7865646e, 0x00545845, 0x00040005, 0x0000002c, 0x69646e49, 0x00736563,
    0x00040006, 0x0000002c, 0x00000000, 0x00000069, 0x00050005, 0x00000037, 0x74726556, 0x73656369, 0x00000000,
    0x00040006, 0x00000037, 0x00000000, 0x00000076, 0x00040005, 0x0000007c, 0x72747461, 0x00736269, 0x00060005,
    0x000000b2, 0x505f6c67, 0x696d6972, 0x65766974, 0x00004449, 0x00050005, 0x000000ba, 0x56746968, 0x65756c61,
    0x00000000, 0x00050005, 0x000000c7, 0x74786574, 0x73657275, 0x00000000, 0x00050005, 0x000000fe, 0x64616873,
    0x6465776f, 0x00000000, 0x00050048, 0x0000001d, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x0000001d,
    0x00000001, 0x00000023, 0x00000008, 0x00050048, 0x0000001d, 0x00000002, 0x00000023, 0x00000010, 0x00050048,
    0x0000001d, 0x00000003, 0x00000023, 0x00000014, 0x00040047, 0x0000001e, 0x00000006, 0x00000018, 0x00030047,
    0x0000001f, 0x00000002, 0x00050048, 0x0000001f, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x00000021,
    0x00000021, 0x00000004, 0x00040047, 0x00000021, 0x00000022, 0x00000000, 0x00040047, 0x00000024, 0x0000000b,
    0x000014e8, 0x00040047, 0x0000002b, 0x00000006, 0x00000004, 0x00030047, 0x0000002c, 0x00000002, 0x00050048,
    0x0000002c, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x00000036, 0x00000006, 0x00000010, 0x00030047,
    0x00000037, 0x00000002, 0x00050048, 0x00000037, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x000000b2,
    0x0000000b, 0x00000007, 0x00040047, 0x000000c7, 0x00000021, 0x00000005, 0x00040047, 0x000000c7, 0x00000022,
    0x00000000, 0x00030047, 0x000000ca, 0x000014b4, 0x00030047, 0x000000cc, 0x000014b4, 0x00030047, 0x000000cd,
    0x000014b4, 0x00030047, 0x000000dd, 0x000014b4, 0x00030047, 0x000000de, 0x000014b4, 0x00030047, 0x000000df,
    0x000014b4, 0x00020013, 0x00000002, 0x00030021, 0x00000003, 0x00000002, 0x00040015, 0x00000006, 0x00000020,
    0x00000000, 0x00040015, 0x00000008, 0x00000020, 0x00000001, 0x00030016, 0x0000000a, 0x00000020, 0x00040017,
    0x0000000b, 0x0000000a, 0x00000003, 0x00040017, 0x0000000c, 0x0000000a, 0x00000002, 0x0005001e, 0x0000000d,
    0x0000000b, 0x0000000b, 0x0000000c, 0x0004002b, 0x00000006, 0x0000000e, 0x00000003, 0x0004001c, 0x0000000f,
    0x0000000d, 0x0000000e, 0x00040015, 0x00000019, 0x00000040, 0x00000000, 0x0006001e, 0x0000001a, 0x00000019,
    0x00000019, 0x00000008, 0x00000008, 0x0006001e, 0x0000001d, 0x00000019, 0x00000019, 0x00000008, 0x00000008,
    0x0003001d, 0x0000001e, 0x0000001d, 0x0003001e, 0x0000001f, 0x0000001e, 0x00040020, 0x00000020, 0x0000000c,
    0x0000001f, 0x0004003b, 0x00000020, 0x00000021, 0x0000000c, 0x0004002b, 0x00000008, 0x00000022, 0x00000000,
    0x00040020, 0x00000023, 0x00000001, 0x00000008, 0x0004003b, 0x00000023, 0x00000024, 0x00000001, 0x00040020,
    0x00000026, 0x0000000c, 0x0000001d, 0x00030027, 0x0000002a, 0x000014e5, 0x0003001d, 0x0000002b, 0x00000006,
    0x0003001e, 0x0000002c, 0x0000002b, 0x00040020, 0x0000002a, 0x000014e5, 0x0000002c, 0x0004002b, 0x00000008,
    0x0000002f, 0x00000001, 0x00030027, 0x00000034, 0x000014e5, 0x00040017, 0x00000035, 0x0000000a, 0x00000004,
    0x0003001d, 0x00000036, 0x00000035, 0x0003001e, 0x00000037, 0x00000036, 0x00040020, 0x00000034, 0x000014e5,
    0x00000037, 0x0004002b, 0x00000006, 0x0000003e, 0x00000000, 0x00020014, 0x00000045, 0x00040020, 0x0000004c,
    0x000014e5, 0x00000006, 0x0004002b, 0x00000006, 0x0000004f, 0x00000006, 0x00040020, 0x00000056, 0x000014e5,
    0x00000035, 0x0004002b, 0x00000006, 0x0000005c, 0x00000001, 0x00040020, 0x00000065, 0x00000007, 0x0000000b,
    0x0004002b, 0x00000008, 0x00000072, 0x00000002, 0x00040020, 0x00000075, 0x00000007, 0x0000000c, 0x0004002b,
    0x0000000a, 0x0000007a, 0x3f800000, 0x00040020, 0x0000007b, 0x000014db, 0x0000000c, 0x0004003b, 0x0000007b,
    0x0000007c, 0x000014db, 0x00040020, 0x0000007d, 0x000014db, 0x0000000a, 0x0004003b, 0x00000023, 0x000000b2,
    0x00000001, 0x00040020, 0x000000b9, 0x000014de, 0x0000000b, 0x0004003b, 0x000000b9, 0x000000ba, 0x000014de,
    0x00090019, 0x000000c3, 0x0000000a, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
    0x0003001b, 0x000000c4, 0x000000c3, 0x0003001d, 0x000000c5, 0x000000c4, 0x00040020, 0x000000c6, 0x00000000,
    0x000000c5, 0x0004003b, 0x000000c6, 0x000000c7, 0x00000000, 0x00040020, 0x000000cb, 0x00000000, 0x000000c4,
    0x0004002b, 0x0000000a, 0x000000d0, 0x00000000, 0x0004002b, 0x00000008, 0x000000d6, 0xffffffff, 0x00040020,
    0x000000fd, 0x000014da, 0x00000045, 0x0004003b, 0x000000fd, 0x000000fe, 0x000014da, 0x00030029, 0x00000045,
    0x000000ff, 0x00040020, 0x00000183, 0x00000007, 0x0000000f, 0x00050036, 0x00000002, 0x00000004, 0x00000000,
    0x00000003, 0x000200f8, 0x00000005, 0x0004003b, 0x00000183, 0x00000184, 0x00000007, 0x0004003d, 0x00000008,
    0x000000b3, 0x000000b2, 0x0004007c, 0x00000006, 0x000000b4, 0x000000b3, 0x00050084, 0x00000006, 0x00000118,
    0x000000b4, 0x0000000e, 0x0004003d, 0x00000008, 0x00000119, 0x00000024, 0x00060041, 0x00000026, 0x0000011a,
    0x00000021, 0x00000022, 0x00000119, 0x0004003d, 0x0000001d, 0x0000011b, 0x0000011a, 0x00040190, 0x0000001a,
    0x0000011c, 0x0000011b, 0x00050051, 0x00000019, 0x00000181, 0x0000011c, 0x00000000, 0x00050051, 0x00000019,
    0x00000182, 0x0000011c, 0x00000001, 0x00040078, 0x0000002a, 0x0000011f, 0x00000182, 0x00040078, 0x00000034,
    0x00000122, 0x00000181, 0x000200f9, 0x00000123, 0x000200f8, 0x00000123, 0x000700f5, 0x00000006, 0x000001d8,
    0x0000003e, 0x00000005, 0x0000014c, 0x00000127, 0x000500b0, 0x00000045, 0x00000126, 0x000001d8, 0x0000000e,
    0x000400f6, 0x0000014d, 0x00000127, 0x00000000, 0x000400fa, 0x00000126, 0x00000127, 0x0000014d, 0x000200f8,
    0x00000127, 0x00050080, 0x00000006, 0x0000012b, 0x00000118, 0x000001d8, 0x00060041, 0x0000004c, 0x0000012c,
    0x0000011f, 0x00000022, 0x0000012b, 0x0006003d, 0x00000006, 0x0000012d, 0x0000012c, 0x00000002, 0x00000004,
    0x00050084, 0x00000006, 0x0000012e, 0x0000012d, 0x0000004f, 0x00060041, 0x00000056, 0x00000132, 0x00000122,
    0x00000022, 0x0000012e, 0x0006003d, 0x00000035, 0x00000133, 0x00000132, 0x00000002, 0x00000010, 0x00050080,
    0x00000006, 0x00000136, 0x0000012e, 0x0000005c, 0x00060041, 0x00000056, 0x00000137, 0x00000122, 0x00000022,
    0x00000136, 0x0006003d, 0x00000035, 0x00000138, 0x00000137, 0x00000002, 0x00000010, 0x0008004f, 0x0000000b,
    0x0000013b, 0x00000133, 0x00000133, 0x00000000, 0x00000001, 0x00000002, 0x00060041, 0x00000065, 0x00000187,
    0x00000184, 0x000001d8, 0x00000022, 0x0003003e, 0x00000187, 0x0000013b, 0x00050051, 0x0000000a, 0x0000013f,
    0x00000133, 0x00000003, 0x00050051, 0x0000000a, 0x00000142, 0x00000138, 0x00000000, 0x00050051, 0x0000000a,
    0x00000143, 0x00000138, 0x00000001, 0x00060050, 0x0000000b, 0x00000144, 0x0000013f, 0x00000142, 0x00000143,
    0x00060041, 0x00000065, 0x00000188, 0x00000184, 0x000001d8, 0x0000002f, 0x0003003e, 0x00000188, 0x00000144,
    0x0007004f, 0x0000000c, 0x00000148, 0x00000138, 0x00000138, 0x00000002, 0x00000003, 0x00060041, 0x00000075,
    0x00000189, 0x00000184, 0x000001d8, 0x00000072, 0x0003003e, 0x00000189, 0x00000148, 0x00050080, 0x00000006,
    0x0000014c, 0x000001d8, 0x0000002f, 0x000200f9, 0x00000123, 0x000200f8, 0x0000014d, 0x00050041, 0x0000007d,
    0x0000014e, 0x0000007c, 0x0000003e, 0x0004003d, 0x0000000a, 0x0000014f, 0x0000014e, 0x00050083, 0x0000000a,
    0x00000150, 0x0000007a, 0x0000014f, 0x00050041, 0x0000007d, 0x00000151, 0x0000007c, 0x0000005c, 0x0004003d,
    0x0000000a, 0x00000152, 0x00000151, 0x00050083, 0x0000000a, 0x00000153, 0x00000150, 0x00000152, 0x0004003d,
    0x0000000a, 0x00000155, 0x0000014e, 0x0004003d, 0x0000000a, 0x00000157, 0x00000151, 0x00060041, 0x00000075,
    0x0000018a, 0x00000184, 0x00000022, 0x00000072, 0x0004003d, 0x0000000c, 0x0000015a, 0x0000018a, 0x0005008e,
    0x0000000c, 0x0000015d, 0x0000015a, 0x00000153, 0x00060041, 0x00000075, 0x0000018b, 0x00000184, 0x0000002f,
    0x00000072, 0x0004003d, 0x0000000c, 0x0000015f, 0x0000018b, 0x0005008e, 0x0000000c, 0x00000162, 0x0000015f,
    0x00000155, 0x00050081, 0x0000000c, 0x00000163, 0x0000015d, 0x00000162, 0x00060041, 0x00000075, 0x0000018c,
    0x00000184, 0x00000072, 0x00000072, 0x0004003d, 0x0000000c, 0x00000165, 0x0000018c, 0x0005008e, 0x0000000c,
    0x00000168, 0x00000165, 0x00000157, 0x00050081, 0x0000000c, 0x00000169, 0x00000163, 0x00000168, 0x00060041,
    0x00000065, 0x0000018d, 0x00000184, 0x00000022, 0x0000002f, 0x0004003d, 0x0000000b, 0x0000016c, 0x0000018d,
    0x0005008e, 0x0000000b, 0x0000016f, 0x0000016c, 0x00000153, 0x00060041, 0x00000065, 0x0000018e, 0x00000184,
    0x0000002f, 0x0000002f, 0x0004003d, 0x0000000b, 0x00000171, 0x0000018e, 0x0005008e, 0x0000000b, 0x00000174,
    0x00000171, 0x00000155, 0x00050081, 0x0000000b, 0x00000175, 0x0000016f, 0x00000174, 0x00060041, 0x00000065,
    0x0000018f, 0x00000184, 0x00000072, 0x0000002f, 0x0004003d, 0x0000000b, 0x00000177, 0x0000018f, 0x0005008e,
    0x0000000b, 0x0000017a, 0x00000177, 0x00000157, 0x00050081, 0x0000000b, 0x0000017b, 0x00000175, 0x0000017a,
    0x0003003e, 0x000000ba, 0x0000017b, 0x0004003d, 0x0000001d, 0x000000c0, 0x0000011a, 0x00040190, 0x0000001a,
    0x000000c1, 0x000000c0, 0x00050051, 0x00000008, 0x000001a6, 0x000000c1, 0x00000002, 0x00050051, 0x00000008,
    0x000001a7, 0x000000c1, 0x00000003, 0x00040053, 0x00000008, 0x000000ca, 0x000001a6, 0x00050041, 0x000000cb,
    0x000000cc, 0x000000c7, 0x000000ca, 0x0004003d, 0x000000c4, 0x000000cd, 0x000000cc, 0x00070058, 0x00000035,
    0x000000d1, 0x000000cd, 0x00000169, 0x00000002, 0x000000d0, 0x0008004f, 0x0000000b, 0x000000d2, 0x000000d1,
    0x000000d1, 0x00000000, 0x00000001, 0x00000002, 0x000500ad, 0x00000045, 0x000000d7, 0x000001a7, 0x000000d6,
    0x000300f7, 0x000000d9, 0x00000000, 0x000400fa, 0x000000d7, 0x000000d8, 0x000000d9, 0x000200f8, 0x000000d8,
    0x00040053, 0x00000008, 0x000000dd, 0x000001a7, 0x00050041, 0x000000cb, 0x000000de, 0x000000c7, 0x000000dd,
    0x0004003d, 0x000000c4, 0x000000df, 0x000000de, 0x00070058, 0x00000035, 0x000000e2, 0x000000df, 0x00000169,
    0x00000002, 0x000000d0, 0x00050051, 0x0000000a, 0x000000e3, 0x000000e2, 0x00000000, 0x0005008e, 0x0000000b,
    0x000000e6, 0x000000d2, 0x000000e3, 0x000200f9, 0x000000d9, 0x000200f8, 0x000000d9, 0x000700f5, 0x0000000b,
    0x000001d9, 0x000000d2, 0x0000014d, 0x000000e6, 0x000000d8, 0x0003003e, 0x000000ba, 0x000001d9, 0x0003003e,
    0x000000fe, 0x000000ff, 0x000100fd, 0x00010038
};

#if 0 // original glsl -> bda_push_constant_offset.comp
#version 450
#extension GL_ARB_gpu_shader_int64 : require
#extension GL_EXT_buffer_reference2 : require
layout(local_size_x = 32, local_size_y = 1, local_size_z = 1) in;

layout(buffer_reference) readonly buffer BDA {
    uint a;
    uint b;
};

// sizeof(address_wrapper_t) == 16
struct address_wrapper_t
{
    uint64_t base_address;
    uint offset;
    uint pad[1];
};

layout(push_constant) uniform PC
{
    // offset 0
    address_wrapper_t address_0;

    // offset 16
    address_wrapper_t address_1;

    uint num_elements;
} pc;

void main() {

    uint gid = gl_GlobalInvocationID.x;
    if(gid >= pc.num_elements){ return; }

    BDA input_ptr = BDA(pc.address_0.base_address + pc.address_0.offset);
    BDA output_ptr = BDA(pc.address_1.base_address + pc.address_1.offset);

    output_ptr[gid].a = input_ptr[gid].a;
    output_ptr[gid].b = input_ptr[gid].b;
}
#endif

const uint32_t spirv_bda_push_constant_offset[] = {
    0x07230203, 0x00010500, 0x0008000b, 0x00000069, 0x00000000, 0x00020011, 0x00000001, 0x00020011, 0x0000000b,
    0x00020011, 0x000014e3, 0x0006000b, 0x00000001, 0x4c534c47, 0x6474732e, 0x3035342e, 0x00000000, 0x0003000e,
    0x000014e4, 0x00000001, 0x0007000f, 0x00000005, 0x00000004, 0x6e69616d, 0x00000000, 0x0000000b, 0x00000017,
    0x00060010, 0x00000004, 0x00000011, 0x00000020, 0x00000001, 0x00000001, 0x00030003, 0x00000002, 0x000001c2,
    0x00070004, 0x415f4c47, 0x675f4252, 0x735f7570, 0x65646168, 0x6e695f72, 0x00343674, 0x00070004, 0x455f4c47,
    0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x0065636e, 0x00080004, 0x455f4c47, 0x625f5458, 0x65666675,
    0x65725f72, 0x65726566, 0x3265636e, 0x00000000, 0x00040005, 0x00000004, 0x6e69616d, 0x00000000, 0x00080005,
    0x0000000b, 0x475f6c67, 0x61626f6c, 0x766e496c, 0x7461636f, 0x496e6f69, 0x00000044, 0x00070005, 0x00000014,
    0x72646461, 0x5f737365, 0x70617277, 0x5f726570, 0x00000074, 0x00070006, 0x00000014, 0x00000000, 0x65736162,
    0x6464615f, 0x73736572, 0x00000000, 0x00050006, 0x00000014, 0x00000001, 0x7366666f, 0x00007465, 0x00040006,
    0x00000014, 0x00000002, 0x00646170, 0x00030005, 0x00000015, 0x00004350, 0x00060006, 0x00000015, 0x00000000,
    0x72646461, 0x5f737365, 0x00000030, 0x00060006, 0x00000015, 0x00000001, 0x72646461, 0x5f737365, 0x00000031,
    0x00070006, 0x00000015, 0x00000002, 0x5f6d756e, 0x6d656c65, 0x73746e65, 0x00000000, 0x00030005, 0x00000017,
    0x00006370, 0x00030005, 0x00000023, 0x00414442, 0x00040006, 0x00000023, 0x00000000, 0x00000061, 0x00040006,
    0x00000023, 0x00000001, 0x00000062, 0x00040047, 0x0000000b, 0x0000000b, 0x0000001c, 0x00040047, 0x00000013,
    0x00000006, 0x00000004, 0x00050048, 0x00000014, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x00000014,
    0x00000001, 0x00000023, 0x00000008, 0x00050048, 0x00000014, 0x00000002, 0x00000023, 0x0000000c, 0x00030047,
    0x00000015, 0x00000002, 0x00050048, 0x00000015, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x00000015,
    0x00000001, 0x00000023, 0x00000010, 0x00050048, 0x00000015, 0x00000002, 0x00000023, 0x00000020, 0x00030047,
    0x00000023, 0x00000002, 0x00040048, 0x00000023, 0x00000000, 0x00000018, 0x00050048, 0x00000023, 0x00000000,
    0x00000023, 0x00000000, 0x00040048, 0x00000023, 0x00000001, 0x00000018, 0x00050048, 0x00000023, 0x00000001,
    0x00000023, 0x00000004, 0x00040047, 0x00000066, 0x0000000b, 0x00000019, 0x00020013, 0x00000002, 0x00030021,
    0x00000003, 0x00000002, 0x00040015, 0x00000006, 0x00000020, 0x00000000, 0x00040017, 0x00000009, 0x00000006,
    0x00000003, 0x00040020, 0x0000000a, 0x00000001, 0x00000009, 0x0004003b, 0x0000000a, 0x0000000b, 0x00000001,
    0x0004002b, 0x00000006, 0x0000000c, 0x00000000, 0x00040020, 0x0000000d, 0x00000001, 0x00000006, 0x00040015,
    0x00000011, 0x00000040, 0x00000000, 0x0004002b, 0x00000006, 0x00000012, 0x00000001, 0x0004001c, 0x00000013,
    0x00000006, 0x00000012, 0x0005001e, 0x00000014, 0x00000011, 0x00000006, 0x00000013, 0x0005001e, 0x00000015,
    0x00000014, 0x00000014, 0x00000006, 0x00040020, 0x00000016, 0x00000009, 0x00000015, 0x0004003b, 0x00000016,
    0x00000017, 0x00000009, 0x00040015, 0x00000018, 0x00000020, 0x00000001, 0x0004002b, 0x00000018, 0x00000019,
    0x00000002, 0x00040020, 0x0000001a, 0x00000009, 0x00000006, 0x00020014, 0x0000001d, 0x00030027, 0x00000022,
    0x000014e5, 0x0004001e, 0x00000023, 0x00000006, 0x00000006, 0x00040020, 0x00000022, 0x000014e5, 0x00000023,
    0x0004002b, 0x00000018, 0x00000026, 0x00000000, 0x00040020, 0x00000027, 0x00000009, 0x00000011, 0x0004002b,
    0x00000018, 0x0000002a, 0x00000001, 0x00040015, 0x0000003b, 0x00000040, 0x00000001, 0x0005002b, 0x00000011,
    0x0000003f, 0x00000010, 0x00000000, 0x00040020, 0x0000004c, 0x000014e5, 0x00000006, 0x0004002b, 0x00000006,
    0x00000065, 0x00000020, 0x0006002c, 0x00000009, 0x00000066, 0x00000065, 0x00000012, 0x00000012, 0x00050036,
    0x00000002, 0x00000004, 0x00000000, 0x00000003, 0x000200f8, 0x00000005, 0x000300f7, 0x00000067, 0x00000000,
    0x000300fb, 0x0000000c, 0x00000068, 0x000200f8, 0x00000068, 0x00050041, 0x0000000d, 0x0000000e, 0x0000000b,
    0x0000000c, 0x0004003d, 0x00000006, 0x0000000f, 0x0000000e, 0x00050041, 0x0000001a, 0x0000001b, 0x00000017,
    0x00000019, 0x0004003d, 0x00000006, 0x0000001c, 0x0000001b, 0x000500ae, 0x0000001d, 0x0000001e, 0x0000000f,
    0x0000001c, 0x000300f7, 0x00000020, 0x00000000, 0x000400fa, 0x0000001e, 0x0000001f, 0x00000020, 0x000200f8,
    0x0000001f, 0x000200f9, 0x00000067, 0x000200f8, 0x00000020, 0x00060041, 0x00000027, 0x00000028, 0x00000017,
    0x00000026, 0x00000026, 0x0004003d, 0x00000011, 0x00000029, 0x00000028, 0x00060041, 0x0000001a, 0x0000002b,
    0x00000017, 0x00000026, 0x0000002a, 0x0004003d, 0x00000006, 0x0000002c, 0x0000002b, 0x00040071, 0x00000011,
    0x0000002d, 0x0000002c, 0x00050080, 0x00000011, 0x0000002e, 0x00000029, 0x0000002d, 0x00040078, 0x00000022,
    0x0000002f, 0x0000002e, 0x00060041, 0x00000027, 0x00000031, 0x00000017, 0x0000002a, 0x00000026, 0x0004003d,
    0x00000011, 0x00000032, 0x00000031, 0x00060041, 0x0000001a, 0x00000033, 0x00000017, 0x0000002a, 0x0000002a,
    0x0004003d, 0x00000006, 0x00000034, 0x00000033, 0x00040071, 0x00000011, 0x00000035, 0x00000034, 0x00050080,
    0x00000011, 0x00000036, 0x00000032, 0x00000035, 0x00040078, 0x00000022, 0x00000037, 0x00000036, 0x00040075,
    0x00000011, 0x00000039, 0x00000037, 0x00040071, 0x00000011, 0x0000003c, 0x0000000f, 0x0004007c, 0x0000003b,
    0x0000003d, 0x0000003c, 0x0004007c, 0x00000011, 0x0000003e, 0x0000003d, 0x00050084, 0x00000011, 0x00000040,
    0x0000003e, 0x0000003f, 0x00050080, 0x00000011, 0x00000041, 0x00000039, 0x00000040, 0x00040078, 0x00000022,
    0x00000042, 0x00000041, 0x00040075, 0x00000011, 0x00000044, 0x0000002f, 0x00050080, 0x00000011, 0x0000004a,
    0x00000044, 0x00000040, 0x00040078, 0x00000022, 0x0000004b, 0x0000004a, 0x00050041, 0x0000004c, 0x0000004d,
    0x0000004b, 0x00000026, 0x0006003d, 0x00000006, 0x0000004e, 0x0000004d, 0x00000002, 0x00000010, 0x00050041,
    0x0000004c, 0x0000004f, 0x00000042, 0x00000026, 0x0005003e, 0x0000004f, 0x0000004e, 0x00000002, 0x00000010,
    0x00040075, 0x00000011, 0x00000051, 0x00000037, 0x00050080, 0x00000011, 0x00000057, 0x00000051, 0x00000040,
    0x00040078, 0x00000022, 0x00000058, 0x00000057, 0x00040075, 0x00000011, 0x0000005a, 0x0000002f, 0x00050080,
    0x00000011, 0x00000060, 0x0000005a, 0x00000040, 0x00040078, 0x00000022, 0x00000061, 0x00000060, 0x00050041,
    0x0000004c, 0x00000062, 0x00000061, 0x0000002a, 0x0006003d, 0x00000006, 0x00000063, 0x00000062, 0x00000002,
    0x00000004, 0x00050041, 0x0000004c, 0x00000064, 0x00000058, 0x0000002a, 0x0005003e, 0x00000064, 0x00000063,
    0x00000002, 0x00000004, 0x000200f9, 0x00000067, 0x000200f8, 0x00000067, 0x000100fd, 0x00010038
};

#if 0 // original glsl -> bda_pointer_chaining.comp
#version 460

#extension GL_EXT_scalar_block_layout : require
#extension GL_EXT_buffer_reference2 : require
#extension GL_EXT_shader_explicit_arithmetic_types : require

struct camera_t
{
    mat4 view_projection;
};

struct mesh_entry_t
{
    mat4 transform;
    uint buffer_index;
    uint num_vertices;
};

struct vertex_t
{
    float x, y, z;
    uint normal;
    uint tangent;
    float16_t u, v;
};

// buffer of mesh_entry
layout(buffer_reference, scalar) readonly buffer MeshEntryBuffer { mesh_entry_t v[]; };

// array of vertex-buffers
layout(buffer_reference, scalar) readonly buffer VertexBuffer { vertex_t v[]; };
layout(buffer_reference, scalar) readonly buffer VertexBufferArray { VertexBuffer v[]; };

layout(binding = 0, set = 0, std140) uniform UBO
{
    camera_t camera;

    MeshEntryBuffer entries; // regular BDA
    VertexBufferArray vertex_buffers; // this BDA holds an array of pointers (vertex_t*[])

    VertexBuffer out_vertices;
} data;

layout(local_size_x = 64, local_size_y = 1, local_size_z = 1) in;

void main()
{
    uint gid = gl_GlobalInvocationID.x;

    // normally fetched from somewhere else
    uint entry_index = 0, vertex_offset = gid;

    // regular BDA-access
    mesh_entry_t entry = data.entries.v[entry_index];
    if(gid >= entry.num_vertices){ return; }

    // pointer-chain
    vertex_t vertex = data.vertex_buffers.v[entry.buffer_index].v[vertex_offset];

    data.out_vertices.v[vertex_offset] = vertex;
}
#endif

const uint32_t spirv_bda_pointer_chaining[] = {
    0x07230203, 0x00010500, 0x0008000b, 0x00000063, 0x00000000, 0x00020011, 0x00000001, 0x00020011, 0x00000009,
    0x00020011, 0x00001151, 0x00020011, 0x000014e3, 0x0006000b, 0x00000001, 0x4c534c47, 0x6474732e, 0x3035342e,
    0x00000000, 0x0003000e, 0x000014e4, 0x00000001, 0x0007000f, 0x00000005, 0x00000004, 0x6e69616d, 0x00000000,
    0x0000000b, 0x00000028, 0x00060010, 0x00000004, 0x00000011, 0x00000040, 0x00000001, 0x00000001, 0x00030003,
    0x00000002, 0x000001cc, 0x00070004, 0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x0065636e,
    0x00080004, 0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x3265636e, 0x00000000, 0x00080004,
    0x455f4c47, 0x735f5458, 0x616c6163, 0x6c625f72, 0x5f6b636f, 0x6f79616c, 0x00007475, 0x000b0004, 0x455f4c47,
    0x735f5458, 0x65646168, 0x78655f72, 0x63696c70, 0x615f7469, 0x68746972, 0x6974656d, 0x79745f63, 0x00736570,
    0x00040005, 0x00000004, 0x6e69616d, 0x00000000, 0x00080005, 0x0000000b, 0x475f6c67, 0x61626f6c, 0x766e496c,
    0x7461636f, 0x496e6f69, 0x00000044, 0x00060005, 0x00000016, 0x6873656d, 0x746e655f, 0x745f7972, 0x00000000,
    0x00060006, 0x00000016, 0x00000000, 0x6e617274, 0x726f6673, 0x0000006d, 0x00070006, 0x00000016, 0x00000001,
    0x66667562, 0x695f7265, 0x7865646e, 0x00000000, 0x00070006, 0x00000016, 0x00000002, 0x5f6d756e, 0x74726576,
    0x73656369, 0x00000000, 0x00050005, 0x00000019, 0x656d6163, 0x745f6172, 0x00000000, 0x00070006, 0x00000019,
    0x00000000, 0x77656976, 0x6f72705f, 0x7463656a, 0x006e6f69, 0x00030005, 0x0000001d, 0x004f4255, 0x00050006,
    0x0000001d, 0x00000000, 0x656d6163, 0x00006172, 0x00050006, 0x0000001d, 0x00000001, 0x72746e65, 0x00736569,
    0x00070006, 0x0000001d, 0x00000002, 0x74726576, 0x625f7865, 0x65666675, 0x00007372, 0x00070006, 0x0000001d,
    0x00000003, 0x5f74756f, 0x74726576, 0x73656369, 0x00000000, 0x00060005, 0x0000001e, 0x6873656d, 0x746e655f,
    0x745f7972, 0x00000000, 0x00060006, 0x0000001e, 0x00000000, 0x6e617274, 0x726f6673, 0x0000006d, 0x00070006,
    0x0000001e, 0x00000001, 0x66667562, 0x695f7265, 0x7865646e, 0x00000000, 0x00070006, 0x0000001e, 0x00000002,
    0x5f6d756e, 0x74726576, 0x73656369, 0x00000000, 0x00060005, 0x00000020, 0x6873654d, 0x72746e45, 0x66754279,
    0x00726566, 0x00040006, 0x00000020, 0x00000000, 0x00000076, 0x00070005, 0x00000022, 0x74726556, 0x75427865,
    0x72656666, 0x61727241, 0x00000079, 0x00040006, 0x00000022, 0x00000000, 0x00000076, 0x00050005, 0x00000024,
    0x74726576, 0x745f7865, 0x00000000, 0x00040006, 0x00000024, 0x00000000, 0x00000078, 0x00040006, 0x00000024,
    0x00000001, 0x00000079, 0x00040006, 0x00000024, 0x00000002, 0x0000007a, 0x00050006, 0x00000024, 0x00000003,
    0x6d726f6e, 0x00006c61, 0x00050006, 0x00000024, 0x00000004, 0x676e6174, 0x00746e65, 0x00040006, 0x00000024,
    0x00000005, 0x00000075, 0x00040006, 0x00000024, 0x00000006, 0x00000076, 0x00060005, 0x00000026, 0x74726556,
    0x75427865, 0x72656666, 0x00000000, 0x00040006, 0x00000026, 0x00000000, 0x00000076, 0x00040005, 0x00000028,
    0x61746164, 0x00000000, 0x00050005, 0x0000003d, 0x74726576, 0x745f7865, 0x00000000, 0x00040006, 0x0000003d,
    0x00000000, 0x00000078, 0x00040006, 0x0000003d, 0x00000001, 0x00000079, 0x00040006, 0x0000003d, 0x00000002,
    0x0000007a, 0x00050006, 0x0000003d, 0x00000003, 0x6d726f6e, 0x00006c61, 0x00050006, 0x0000003d, 0x00000004,
    0x676e6174, 0x00746e65, 0x00040006, 0x0000003d, 0x00000005, 0x00000075, 0x00040006, 0x0000003d, 0x00000006,
    0x00000076, 0x00040047, 0x0000000b, 0x0000000b, 0x0000001c, 0x00040048, 0x00000019, 0x00000000, 0x00000005,
    0x00050048, 0x00000019, 0x00000000, 0x00000007, 0x00000010, 0x00050048, 0x00000019, 0x00000000, 0x00000023,
    0x00000000, 0x00030047, 0x0000001d, 0x00000002, 0x00050048, 0x0000001d, 0x00000000, 0x00000023, 0x00000000,
    0x00050048, 0x0000001d, 0x00000001, 0x00000023, 0x00000040, 0x00050048, 0x0000001d, 0x00000002, 0x00000023,
    0x00000048, 0x00050048, 0x0000001d, 0x00000003, 0x00000023, 0x00000050, 0x00040048, 0x0000001e, 0x00000000,
    0x00000005, 0x00050048, 0x0000001e, 0x00000000, 0x00000007, 0x00000010, 0x00050048, 0x0000001e, 0x00000000,
    0x00000023, 0x00000000, 0x00050048, 0x0000001e, 0x00000001, 0x00000023, 0x00000040, 0x00050048, 0x0000001e,
    0x00000002, 0x00000023, 0x00000044, 0x00040047, 0x0000001f, 0x00000006, 0x00000048, 0x00030047, 0x00000020,
    0x00000002, 0x00040048, 0x00000020, 0x00000000, 0x00000018, 0x00050048, 0x00000020, 0x00000000, 0x00000023,
    0x00000000, 0x00040047, 0x00000021, 0x00000006, 0x00000008, 0x00030047, 0x00000022, 0x00000002, 0x00040048,
    0x00000022, 0x00000000, 0x00000018, 0x00050048, 0x00000022, 0x00000000, 0x00000023, 0x00000000, 0x00050048,
    0x00000024, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x00000024, 0x00000001, 0x00000023, 0x00000004,
    0x00050048, 0x00000024, 0x00000002, 0x00000023, 0x00000008, 0x00050048, 0x00000024, 0x00000003, 0x00000023,
    0x0000000c, 0x00050048, 0x00000024, 0x00000004, 0x00000023, 0x00000010, 0x00050048, 0x00000024, 0x00000005,
    0x00000023, 0x00000014, 0x00050048, 0x00000024, 0x00000006, 0x00000023, 0x00000016, 0x00040047, 0x00000025,
    0x00000006, 0x00000018, 0x00030047, 0x00000026, 0x00000002, 0x00040048, 0x00000026, 0x00000000, 0x00000018,
    0x00050048, 0x00000026, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x00000028, 0x00000021, 0x00000000,
    0x00040047, 0x00000028, 0x00000022, 0x00000000, 0x00040047, 0x00000057, 0x0000000b, 0x00000019, 0x00020013,
    0x00000002, 0x00030021, 0x00000003, 0x00000002, 0x00040015, 0x00000006, 0x00000020, 0x00000000, 0x00040017,
    0x00000009, 0x00000006, 0x00000003, 0x00040020, 0x0000000a, 0x00000001, 0x00000009, 0x0004003b, 0x0000000a,
    0x0000000b, 0x00000001, 0x0004002b, 0x00000006, 0x0000000c, 0x00000000, 0x00040020, 0x0000000d, 0x00000001,
    0x00000006, 0x00030016, 0x00000013, 0x00000020, 0x00040017, 0x00000014, 0x00000013, 0x00000004, 0x00040018,
    0x00000015, 0x00000014, 0x00000004, 0x0005001e, 0x00000016, 0x00000015, 0x00000006, 0x00000006, 0x0003001e,
    0x00000019, 0x00000015, 0x00030027, 0x0000001a, 0x000014e5, 0x00030027, 0x0000001b, 0x000014e5, 0x00030027,
    0x0000001c, 0x000014e5, 0x0006001e, 0x0000001d, 0x00000019, 0x0000001a, 0x0000001b, 0x0000001c, 0x0005001e,
    0x0000001e, 0x00000015, 0x00000006, 0x00000006, 0x0003001d, 0x0000001f, 0x0000001e, 0x0003001e, 0x00000020,
    0x0000001f, 0x00040020, 0x0000001a, 0x000014e5, 0x00000020, 0x0003001d, 0x00000021, 0x0000001c, 0x0003001e,
    0x00000022, 0x00000021, 0x00040020, 0x0000001b, 0x000014e5, 0x00000022, 0x00030016, 0x00000023, 0x00000010,
    0x0009001e, 0x00000024, 0x00000013, 0x00000013, 0x00000013, 0x00000006, 0x00000006, 0x00000023, 0x00000023,
    0x0003001d, 0x00000025, 0x00000024, 0x0003001e, 0x00000026, 0x00000025, 0x00040020, 0x0000001c, 0x000014e5,
    0x00000026, 0x00040020, 0x00000027, 0x00000002, 0x0000001d, 0x0004003b, 0x00000027, 0x00000028, 0x00000002,
    0x00040015, 0x00000029, 0x00000020, 0x00000001, 0x0004002b, 0x00000029, 0x0000002a, 0x00000001, 0x00040020,
    0x0000002b, 0x00000002, 0x0000001a, 0x0004002b, 0x00000029, 0x0000002e, 0x00000000, 0x00040020, 0x00000030,
    0x000014e5, 0x0000001e, 0x0004002b, 0x00000029, 0x00000035, 0x00000002, 0x00020014, 0x00000038, 0x0009001e,
    0x0000003d, 0x00000013, 0x00000013, 0x00000013, 0x00000006, 0x00000006, 0x00000023, 0x00000023, 0x00040020,
    0x00000040, 0x00000002, 0x0000001b, 0x00040020, 0x00000045, 0x000014e5, 0x0000001c, 0x00040020, 0x00000049,
    0x000014e5, 0x00000024, 0x0004002b, 0x00000029, 0x0000004d, 0x00000003, 0x00040020, 0x0000004e, 0x00000002,
    0x0000001c, 0x0004002b, 0x00000006, 0x00000055, 0x00000040, 0x0004002b, 0x00000006, 0x00000056, 0x00000001,
    0x0006002c, 0x00000009, 0x00000057, 0x00000055, 0x00000056, 0x00000056, 0x00050036, 0x00000002, 0x00000004,
    0x00000000, 0x00000003, 0x000200f8, 0x00000005, 0x000300f7, 0x00000058, 0x00000000, 0x000300fb, 0x0000000c,
    0x00000059, 0x000200f8, 0x00000059, 0x00050041, 0x0000000d, 0x0000000e, 0x0000000b, 0x0000000c, 0x0004003d,
    0x00000006, 0x0000000f, 0x0000000e, 0x00050041, 0x0000002b, 0x0000002c, 0x00000028, 0x0000002a, 0x0004003d,
    0x0000001a, 0x0000002d, 0x0000002c, 0x00060041, 0x00000030, 0x00000031, 0x0000002d, 0x0000002e, 0x0000000c,
    0x0006003d, 0x0000001e, 0x00000032, 0x00000031, 0x00000002, 0x00000008, 0x00040190, 0x00000016, 0x00000033,
    0x00000032, 0x00050051, 0x00000006, 0x00000061, 0x00000033, 0x00000001, 0x00050051, 0x00000006, 0x00000062,
    0x00000033, 0x00000002, 0x000500ae, 0x00000038, 0x00000039, 0x0000000f, 0x00000062, 0x000300f7, 0x0000003b,
    0x00000000, 0x000400fa, 0x00000039, 0x0000003a, 0x0000003b, 0x000200f8, 0x0000003a, 0x000200f9, 0x00000058,
    0x000200f8, 0x0000003b, 0x00050041, 0x00000040, 0x00000041, 0x00000028, 0x00000035, 0x0004003d, 0x0000001b,
    0x00000042, 0x00000041, 0x00060041, 0x00000045, 0x00000046, 0x00000042, 0x0000002e, 0x00000061, 0x0006003d,
    0x0000001c, 0x00000047, 0x00000046, 0x00000002, 0x00000008, 0x00060041, 0x00000049, 0x0000004a, 0x00000047,
    0x0000002e, 0x0000000f, 0x0006003d, 0x00000024, 0x0000004b, 0x0000004a, 0x00000002, 0x00000008, 0x00040190,
    0x0000003d, 0x0000004c, 0x0000004b, 0x00050041, 0x0000004e, 0x0000004f, 0x00000028, 0x0000004d, 0x0004003d,
    0x0000001c, 0x00000050, 0x0000004f, 0x00060041, 0x00000049, 0x00000053, 0x00000050, 0x0000002e, 0x0000000f,
    0x00040190, 0x00000024, 0x00000054, 0x0000004c, 0x0005003e, 0x00000053, 0x00000054, 0x00000002, 0x00000008,
    0x000200f9, 0x00000058, 0x000200f8, 0x00000058, 0x000100fd, 0x00010038
};

TEST_CASE("SpirVParsingUtil", "[buffer_references]")
{
    // setting this to kDebugSeverity yields debug-output from parser
    gfxrecon::util::Log::Init(gfxrecon::util::Log::kErrorSeverity);

    gfxrecon::util::SpirVParsingUtil spirVParsingUtil;

    // parsing nullptr yields false
    REQUIRE(!spirVParsingUtil.ParseBufferReferences(nullptr, 0));

    REQUIRE(spirVParsingUtil.ParseBufferReferences(spirv_bda_array, sizeof(spirv_bda_array)));

    // there are 2 buffer-references in a storage-buffer, pointing at an array of 'geometry-nodes'
    auto buffer_references = spirVParsingUtil.GetBufferReferenceInfos();
    REQUIRE(buffer_references.size() == 2);
    REQUIRE(buffer_references[0].source == gfxrecon::util::SpirVParsingUtil::BufferReferenceLocation::STORAGE_BUFFER);
    REQUIRE(buffer_references[0].set == 0);
    REQUIRE(buffer_references[0].binding == 4);
    REQUIRE(buffer_references[0].buffer_offset == 0);
    REQUIRE(buffer_references[0].array_stride == 24);

    REQUIRE(buffer_references[1].source == gfxrecon::util::SpirVParsingUtil::BufferReferenceLocation::STORAGE_BUFFER);
    REQUIRE(buffer_references[1].set == 0);
    REQUIRE(buffer_references[1].binding == 4);
    REQUIRE(buffer_references[1].buffer_offset == 8);
    REQUIRE(buffer_references[1].array_stride == 24);

    gfxrecon::util::Log::Release();
}

TEST_CASE("SpirVParsingUtil_offsets", "[buffer_references_with_offset]")
{
    // setting this to kDebugSeverity yields debug-output from parser
    gfxrecon::util::Log::Init(gfxrecon::util::Log::kErrorSeverity);

    gfxrecon::util::SpirVParsingUtil spirVParsingUtil;

    REQUIRE(
        spirVParsingUtil.ParseBufferReferences(spirv_bda_push_constant_offset, sizeof(spirv_bda_push_constant_offset)));

    auto buffer_references = spirVParsingUtil.GetBufferReferenceInfos();

    // define what to expect
    REQUIRE(buffer_references.size() == 2);
    REQUIRE(buffer_references[0].source ==
            gfxrecon::util::SpirVParsingUtil::BufferReferenceLocation::PUSH_CONSTANT_BLOCK);
    REQUIRE(buffer_references[0].buffer_offset == 0);
    REQUIRE(buffer_references[1].source ==
            gfxrecon::util::SpirVParsingUtil::BufferReferenceLocation::PUSH_CONSTANT_BLOCK);

    // check correct struct-offset calculation
    REQUIRE(buffer_references[1].buffer_offset == 16);

    gfxrecon::util::Log::Release();
}

TEST_CASE("SpirVParsingUtil_pointer_chaining", "[buffer_references_pointer_chaining]")
{
    // setting this to kDebugSeverity yields debug-output from parser
    gfxrecon::util::Log::Init(gfxrecon::util::Log::kErrorSeverity);

    gfxrecon::util::SpirVParsingUtil spirVParsingUtil;

    REQUIRE(spirVParsingUtil.ParseBufferReferences(spirv_bda_pointer_chaining, sizeof(spirv_bda_pointer_chaining)));

    auto buffer_references = spirVParsingUtil.GetBufferReferenceInfos();

    // define what to expect
    REQUIRE(buffer_references.size() == 4);

    // MeshEntryBuffer entries; // regular BDA
    REQUIRE(buffer_references[0].source == gfxrecon::util::SpirVParsingUtil::BufferReferenceLocation::UNIFORM_BUFFER);
    REQUIRE(buffer_references[0].buffer_offset == 64);

    // VertexBufferArray vertex_buffers; -> 1st access
    REQUIRE(buffer_references[1].source == gfxrecon::util::SpirVParsingUtil::BufferReferenceLocation::UNIFORM_BUFFER);
    REQUIRE(buffer_references[1].set == 0);
    REQUIRE(buffer_references[1].binding == 0);
    REQUIRE(buffer_references[1].buffer_offset == 72);
    REQUIRE(buffer_references[1].array_stride == 0);

    // NOTE: we identify chained accesses by similar set/binding/offset
    // VertexBufferArray vertex_buffers; -> 2nd access
    REQUIRE(buffer_references[2].source == buffer_references[1].source);
    REQUIRE(buffer_references[2].set == buffer_references[1].set);
    REQUIRE(buffer_references[2].binding == buffer_references[1].binding);
    REQUIRE(buffer_references[2].buffer_offset == buffer_references[1].buffer_offset);
    REQUIRE(buffer_references[2].array_stride == 8);

    // VertexBuffer out_vertices;
    REQUIRE(buffer_references[3].source == gfxrecon::util::SpirVParsingUtil::BufferReferenceLocation::UNIFORM_BUFFER);
    REQUIRE(buffer_references[3].buffer_offset == 80);
    REQUIRE(buffer_references[3].array_stride == 0);

    gfxrecon::util::Log::Release();
}