#!/usr/bin/env bash

# Script to generate a CMake fragment listing all tests in the JSON test suite
# This allows registering tests even before the test data is downloaded

set -e

# Default values
JSONTESTSUITEURL="${JSONTESTSUITEURL:-https://github.com/nst/JSONTestSuite}"
JSONTESTSUITECHECKOUT="${JSONTESTSUITECHECKOUT:-d64aefb55228d9584d3e5b2433f720ea8fd00c82}"
OUTPUT_FILE="${1:-jsontestsuite-tests.cmake}"

usage() {
    cat << EOF
Usage: $0 [OUTPUT_FILE]

Generate a CMake fragment listing all tests in the JSON test suite.

Arguments:
    OUTPUT_FILE     Output file path (default: jsontestsuite-tests.cmake)

Environment variables:
    JSONTESTSUITEURL        URL of the test suite repository
    JSONTESTSUITECHECKOUT   Git commit/tag to use

Example:
    $0 cmake/jsontestsuite-tests.cmake
EOF
    exit 1
}

if [ "$1" = "-h" ] || [ "$1" = "--help" ]; then
    usage
fi

echo "Generating jsontestsuite list from ${JSONTESTSUITEURL} @ ${JSONTESTSUITECHECKOUT}"

# Create a temporary directory
TMPDIR=$(mktemp -d)
trap "rm -rf $TMPDIR" EXIT

# Check if we have a local json-test-suite-data directory to use
LOCAL_DATA=""
for dir in "build/test/json-test-suite-data/test_parsing" "json-test-suite-data/test_parsing" "../build/test/json-test-suite-data/test_parsing"; do
    if [ -d "$dir" ]; then
        LOCAL_DATA="$dir"
        echo "Using local test data from: $LOCAL_DATA"
        break
    fi
done

if [ -n "$LOCAL_DATA" ]; then
    # Use local data
    find "$LOCAL_DATA" -maxdepth 1 -type f -name 'y_*.json' | \
        xargs -n1 basename | sort > "$TMPDIR/pass_tests.txt"
    find "$LOCAL_DATA" -maxdepth 1 -type f -name 'n_*.json' | \
        xargs -n1 basename | sort > "$TMPDIR/fail_tests.txt"
    find "$LOCAL_DATA" -maxdepth 1 -type f -name 'i_*.json' | \
        xargs -n1 basename | sort > "$TMPDIR/impl_tests.txt"
else
    # Clone from git
    echo "No local data found, cloning from git repository..."
    cd "$TMPDIR"
    git clone --depth 1 --filter=blob:none --sparse "${JSONTESTSUITEURL}" repo
    cd repo
    git checkout "${JSONTESTSUITECHECKOUT}"
    git sparse-checkout set test_parsing
    git sparse-checkout disable

    # List test files
    find test_parsing -maxdepth 1 -type f -name 'y_*.json' | \
        xargs -n1 basename | sort > "$TMPDIR/pass_tests.txt"
    find test_parsing -maxdepth 1 -type f -name 'n_*.json' | \
        xargs -n1 basename | sort > "$TMPDIR/fail_tests.txt"
    find test_parsing -maxdepth 1 -type f -name 'i_*.json' | \
        xargs -n1 basename | sort > "$TMPDIR/impl_tests.txt"
fi

# Read test counts
PASS_COUNT=$(wc -l < "$TMPDIR/pass_tests.txt")
FAIL_COUNT=$(wc -l < "$TMPDIR/fail_tests.txt")
IMPL_COUNT=$(wc -l < "$TMPDIR/impl_tests.txt")
TOTAL_COUNT=$((PASS_COUNT + FAIL_COUNT + IMPL_COUNT))

echo "Found $PASS_COUNT tests expected to pass (y_*.json)"
echo "Found $FAIL_COUNT tests expected to fail (n_*.json)"
echo "Found $IMPL_COUNT implementation-defined tests (i_*.json)"
echo "Total: $TOTAL_COUNT tests"

# Generate CMake fragment
cat > "$OUTPUT_FILE" << 'EOF'
# Auto-generated file - do not edit manually
# Generated by cmake/generate-jsontestsuite-list.sh
#
# This file contains lists of tests from the JSON test suite
# It allows registering individual tests without requiring the test data
# to be downloaded first.

EOF

echo "# Test suite checkout: ${JSONTESTSUITECHECKOUT}" >> "$OUTPUT_FILE"
echo "# Generated on: $(date -u +"%Y-%m-%d %H:%M:%S UTC")" >> "$OUTPUT_FILE"
echo "# Pass tests: ${PASS_COUNT}" >> "$OUTPUT_FILE"
echo "# Fail tests: ${FAIL_COUNT}" >> "$OUTPUT_FILE"
echo "# Impl tests: ${IMPL_COUNT}" >> "$OUTPUT_FILE"
echo "" >> "$OUTPUT_FILE"

# Write pass tests
echo "# Tests expected to pass (y_*.json)" >> "$OUTPUT_FILE"
echo "set(JSONTESTSUITE_PASS_TESTS" >> "$OUTPUT_FILE"
while IFS= read -r test; do
    echo "    \"$test\"" >> "$OUTPUT_FILE"
done < "$TMPDIR/pass_tests.txt"
echo ")" >> "$OUTPUT_FILE"
echo "" >> "$OUTPUT_FILE"

# Write fail tests
echo "# Tests expected to fail (n_*.json)" >> "$OUTPUT_FILE"
echo "set(JSONTESTSUITE_FAIL_TESTS" >> "$OUTPUT_FILE"
while IFS= read -r test; do
    echo "    \"$test\"" >> "$OUTPUT_FILE"
done < "$TMPDIR/fail_tests.txt"
echo ")" >> "$OUTPUT_FILE"
echo "" >> "$OUTPUT_FILE"

# Write impl tests
echo "# Implementation-defined tests (i_*.json)" >> "$OUTPUT_FILE"
echo "set(JSONTESTSUITE_IMPL_TESTS" >> "$OUTPUT_FILE"
while IFS= read -r test; do
    echo "    \"$test\"" >> "$OUTPUT_FILE"
done < "$TMPDIR/impl_tests.txt"
echo ")" >> "$OUTPUT_FILE"

echo "Generated $OUTPUT_FILE successfully!"
echo "Total tests: $TOTAL_COUNT"
