// SPDX-License-Identifier: LGPL-2.1-or-later
/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2023  NXP Semiconductors. All rights reserved.
 *
 *  This file contains only VOCS related Unit Test cases.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <fcntl.h>

#include <glib.h>

#include "lib/bluetooth.h"
#include "lib/uuid.h"
#include "src/shared/util.h"
#include "src/shared/tester.h"
#include "src/shared/queue.h"
#include "src/shared/att.h"
#include "src/shared/gatt-db.h"
#include "src/shared/gatt-client.h"
#include "src/shared/gatt-server.h"
#include "src/shared/vcp.h"

struct test_data {
	struct gatt_db *db;
	struct bt_vcp *vcp;
	struct bt_gatt_server *server;
	struct queue *ccc_states;
	size_t iovcnt;
	struct iovec *iov;
};

struct notify {
	uint16_t handle, ccc_handle;
	uint8_t *value;
	uint16_t len;
	bt_gatt_server_conf_func_t conf;
	void *user_data;
};

struct ccc_state {
	uint16_t handle;
	uint16_t value;
};

#define iov_data(args...) ((const struct iovec[]) { args })

#define define_test(name, function, _cfg, args...)		\
	do {							\
		const struct iovec iov[] = { args };		\
		static struct test_data data;			\
		data.iovcnt = ARRAY_SIZE(iov_data(args));	\
		data.iov = util_iov_dup(iov, ARRAY_SIZE(iov_data(args))); \
		tester_add(name, &data, NULL, function,	\
				test_teardown);			\
	} while (0)

static void test_complete_cb(const void *user_data)
{
	tester_test_passed();
}

static void print_debug(const char *str, void *user_data)
{
	const char *prefix = user_data;

	if (tester_use_debug())
		tester_debug("%s%s", prefix, str);
}

static void test_teardown(const void *user_data)
{
	struct test_data *data = (void *)user_data;

	bt_vcp_unref(data->vcp);
	bt_gatt_server_unref(data->server);
	util_iov_free(data->iov, data->iovcnt);

	gatt_db_unref(data->db);

	queue_destroy(data->ccc_states, free);

	tester_teardown_complete();
}

static bool ccc_state_match(const void *a, const void *b)
{
	const struct ccc_state *ccc = a;
	uint16_t handle = PTR_TO_UINT(b);

	return ccc->handle == handle;
}

static struct ccc_state *find_ccc_state(struct test_data *data,
				uint16_t handle)
{
	return queue_find(data->ccc_states, ccc_state_match,
				UINT_TO_PTR(handle));
}

static struct ccc_state *get_ccc_state(struct test_data *data,
					uint16_t handle)
{
	struct ccc_state *ccc;

	ccc = find_ccc_state(data, handle);
	if (ccc)
		return ccc;

	ccc = new0(struct ccc_state, 1);
	ccc->handle = handle;
	queue_push_tail(data->ccc_states, ccc);

	return ccc;
}

static void gatt_notify_cb(struct gatt_db_attribute *attrib,
					struct gatt_db_attribute *ccc,
					const uint8_t *value, size_t len,
					struct bt_att *att, void *user_data)
{
	struct test_data *data = user_data;
	uint16_t handle = gatt_db_attribute_get_handle(attrib);

	if (!bt_gatt_server_send_notification(data->server,
			handle, value, len, false))
		printf("%s: Failed to send notification\n", __func__);
}

static void gatt_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offest,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct test_data *data = user_data;
	struct ccc_state *ccc;
	uint16_t handle;
	uint8_t ecode = 0;
	uint16_t value = 0;

	handle = gatt_db_attribute_get_handle(attrib);

	ccc = get_ccc_state(data, handle);
	if (!ccc) {
		ecode = BT_ATT_ERROR_UNLIKELY;
		goto done;
	}

	value = cpu_to_le16(ccc->value);

done:
	gatt_db_attribute_read_result(attrib, id, ecode, (void *)&value,
							sizeof(value));
}

static void test_server(const void *user_data)
{
	struct test_data *data = (void *)user_data;
	struct bt_att *att;
	struct io *io;

	io = tester_setup_io(data->iov, data->iovcnt);
	g_assert(io);

	tester_io_set_complete_func(test_complete_cb);

	att = bt_att_new(io_get_fd(io), false);
	g_assert(att);

	bt_att_set_debug(att, BT_ATT_DEBUG, print_debug, "bt_att:", NULL);

	data->db = gatt_db_new();
	g_assert(data->db);

	gatt_db_ccc_register(data->db, gatt_ccc_read_cb, NULL,
					gatt_notify_cb, data);

	data->vcp = bt_vcp_new(data->db, NULL);
	g_assert(data->vcp);

	data->server = bt_gatt_server_new(data->db, att, 64, 0);
	g_assert(data->server);

	bt_gatt_server_set_debug(data->server, print_debug, "bt_gatt_server",
						NULL);

	data->ccc_states = queue_new();

	tester_io_send();

	bt_att_unref(att);
}

 /* ATT: Exchange MTU Request (0x02) len 2
  *     Client RX MTU: 64
  * ATT: Exchange MTU Response (0x03) len 2
  *     Server RX MTU: 64
  */
#define VOCS_EXCHANGE_MTU \
	IOV_DATA(0x02, 0x40, 0x00), \
	IOV_DATA(0x03, 0x40, 0x00)

 /*  ATT: Read By Group Type Request (0x10) len 6
  *      Handle range: 0x0001-0xffff
  *      Attribute group type: Primary Service (0x2800)
  *
  *  ATT: Read By Group Type Response (0x11) len 7
  *      Attribute data length: 6
  *      Attribute group list: 1 entry
  *      Handle range: 0x000d-0x0016
  *      UUID: Volume Control (0x1844)
  *
  *  ATT: Read By Group Type Request (0x10) len 6
  *      Handle range: 0x0017-0xffff
  *      Attribute group type: Primary Service (0x2800)
  *
  *  ATT: Error Response (0x01) len 4
  *      Read By Group Type Request (0x10)
  *      Handle: 0x0017
  *      Error: Attribute Not Found (0x0a)
  */
#define VOCS_PRIMARY_SERVICE_VCS \
	IOV_DATA(0x10, 0x01, 0x00, 0xff, 0xff, 0x00, 0x28), \
	IOV_DATA(0x11, 0x06, 0x0d, 0x00, 0x16, 0x00, 0x44, 0x18), \
	IOV_DATA(0x10, 0x17, 0x00, 0xff, 0xff, 0x00, 0x28), \
	IOV_DATA(0x01, 0x10, 0x17, 0x00, 0x0a)

 /* ATT: Read By Group Type Request (0x10) len 6
  *      Handle range: 0x0001-0xffff
  *      Attribute group type: Secondary Service (0x2801)
  *
  * ATT: Read By Group Type Response (0x11) len 7
  *     Attribute data length: 6
  *     Attribute group list: 1 entry
  *     Handle range: 0x0001-0x000c
  *     UUID: Volume Offset Control (0x1845)
  *
  * ATT: Read By Group Type Request (0x10) len 6
  *     Handle range: 0x000d-0xffff
  *     Attribute group type: Secondary Service (0x2801)
  *
  * ATT: Error Response (0x01) len 4
  *     Read By Group Type Request (0x10)
  *     Handle: 0x000d
  *     Error: Attribute Not Found (0x0a)
  */
#define VOCS_SECONDARY_SERVICE_VOCS \
	IOV_DATA(0x10, 0x01, 0x00, 0xff, 0xff, 0x01, 0x28), \
	IOV_DATA(0x11, 0x06, 0x01, 0x00, 0x0c, 0x00, 0x45, 0x18), \
	IOV_DATA(0x10, 0x0d, 0x00, 0xff, 0xff, 0x01, 0x28), \
	IOV_DATA(0x01, 0x10, 0x0d, 0x00, 0x0a)

 /* ATT: Read By Type Request (0x08) len 6
  *     Handle range: 0x0001-0xffff
  *     Attribute type: Include (0x2802)
  *
  * ATT: Read By Type Response (0x09) len 9
  *     Attribute data length: 8
  *     Attribute data list: 1 entry
  *     Handle: 0x000e
  *     Value: 01000c004518
  *
  * ATT: Read By Type Request (0x08) len 6
  *     Handle range: 0x000f-0xffff
  *     Attribute type: Include (0x2802)
  *
  * ATT: Error Response (0x01) len 4
  *     Read By Type Request (0x08)
  *     Handle: 0x000f
  *     Error: Attribute Not Found (0x0a)
  */
#define VOCS_INCLUDED_SERVICE_VOCS \
	IOV_DATA(0x08, 0x01, 0x00, 0xff, 0xff, 0x02, 0x28), \
	IOV_DATA(0x09, 0x08, \
		0x0e, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x45, 0x18), \
	IOV_DATA(0x08, 0x0f, 0x00, 0xff, 0xff, 0x02, 0x28), \
	IOV_DATA(0x01, 0x08, 0x0f, 0x00, 0x0a)

 /* ATT: Read By Type Request (0x08) len 6
  *     Handle range: 0x0001-0x000c
  *     Attribute type: Characteristic (0x2803)
  *
  * ATT: Read By Type Response (0x09) len 29
  *     Attribute data length: 7
  *     Attribute data list: 4 entries
  *     Handle: 0x0002
  *     Value: 120300802b
  *     Handle: 0x0005
  *     Value: 120600812b
  *     Handle: 0x0008
  *     Value: 080900822b
  *     Handle: 0x000a
  *     Value: 120b00832b
  *
  * ATT: Read By Type Request (0x08) len 6
  *     Handle range: 0x000b-0x000c
  *     Attribute type: Characteristic (0x2803)
  *
  * ATT: Error Response (0x01) len 4
  *     Read By Type Request (0x08)
  *     Handle: 0x000b
  *     Error: Attribute Not Found (0x0a)
  */
#define VOCS_DISC_CHAR \
	IOV_DATA(0x08, 0x01, 0x00, 0x0c, 0x00, 0x03, 0x28), \
	IOV_DATA(0x09, 0x07, \
		0x02, 0x00, 0x12, 0x03, 0x00, 0x80, 0x2b, \
		0x05, 0x00, 0x12, 0x06, 0x00, 0x81, 0x2b, \
		0x08, 0x00, 0x08, 0x09, 0x00, 0x82, 0x2b, \
		0x0a, 0x00, 0x12, 0x0b, 0x00, 0x83, 0x2b), \
	IOV_DATA(0x08, 0x0b, 0x00, 0x0c, 0x00, 0x03, 0x28), \
	IOV_DATA(0x01, 0x08, 0x0b, 0x00, 0x0a)

 /* ATT: Find Information Request (0x04) len 4
  *    Handle range: 0x0004-0x0004
  *
  * ATT: Find Information Response (0x05) len 5
  *    Format: UUID-16 (0x01)
  *    Handle: 0x0004
  *    UUID: Client Characteristic Configuration (0x2902)
  *
  * ATT: Find Information Request (0x04) len 4
  *    Handle range: 0x0007-0x0007
  *
  * ATT: Find Information Response (0x05) len 5
  *    Format: UUID-16 (0x01)
  *    Handle: 0x0007
  *    UUID: Client Characteristic Configuration (0x2902)
  *
  * ATT: Find Information Request (0x04) len 4
  *    Handle range: 0x000c-0x000c
  *
  * ATT: Find Information Response (0x05) len 5
  *    Format: UUID-16 (0x01)
  *    Handle: 0x000c
  *    UUID: Client Characteristic Configuration (0x2902)
  */
#define VOCS_DISC_CHAR_DESC \
	IOV_DATA(0x04, 0x04, 0x00, 0x04, 0x00), \
	IOV_DATA(0x05, 0x01, 0x04, 0x00, 0x02, 0x29), \
	IOV_DATA(0x04, 0x07, 0x00, 0x07, 0x00), \
	IOV_DATA(0x05, 0x01, 0x07, 0x00, 0x02, 0x29), \
	IOV_DATA(0x04, 0x0c, 0x00, 0x0c, 0x00), \
	IOV_DATA(0x05, 0x01, 0x0c, 0x00, 0x02, 0x29)

 /* ATT: Read Request (0x0a) len 2
  *     Handle: 0x0004
  *
  * ATT: Read Response (0x0b) len 2
  *
  *  ATT: Read Request (0x0a) len 2
  *     Handle: 0x0007
  *
  * ATT: Read Response (0x0b) len 2
  *
  * ATT: Read Request (0x0a) len 2
  *     Handle: 0x000c
  *
  * ATT: Read Request (0x0a) len 2
  *     Handle: 0x000c
  */
#define VOCS_READ_CHAR_DESC \
	IOV_DATA(0x0a, 0x04, 0x00), \
	IOV_DATA(0x0b, 0x00, 0x00), \
	IOV_DATA(0x0a, 0x07, 0x00), \
	IOV_DATA(0x0b, 0x00, 0x00), \
	IOV_DATA(0x0a, 0x0c, 0x00), \
	IOV_DATA(0x0b, 0x00, 0x00)

 /* ATT: Read Request (0x0a) len 2
  *     Handle: 0x0003
  *
  * ATT: Read Response (0x0b) len 3
  */
#define VOCS_READ_CHAR_VOL_OFFSET \
	IOV_DATA(0x0a, 0x03, 0x00), \
	IOV_DATA(0x0b, 0x00, 0x00, 0x00)

 /* ATT: Read By Type Request (0x08) len 6
  *     Handle range: 0x0002-0x0003
  *     Attribute type: Offset State (0x2b80)
  *
  * ATT: Read By Type Response (0x09) len 6
  *     Attribute data length: 5
  *     Attribute data list: 1 entry
  *     Handle: 0x0003
  *     Value: 000000
  */
#define VOCS_READ_CHAR_VOL_OFFSET_UUID \
	IOV_DATA(0x08, 0x02, 0x00, 0x03, 0x00, 0x80, 0x2b), \
	IOV_DATA(0x09, 0x05, 0x03, 0x00, 0x00, 0x00, 0x00)

 /* ATT: Read Request (0x0a) len 2
  *     Handle: 0x0006
  *
  * ATT: Read Response (0x0b) len 4
  */
#define VOCS_READ_CHAR_AUD_LOC \
	IOV_DATA(0x0a, 0x06, 0x00), \
	IOV_DATA(0x0b, 0x02, 0x00, 0x00, 0x00)

 /* ATT: Read By Type Request (0x08) len 6
  *     Handle range: 0x0005-0x0006
  *     Attribute type: Audio Location (0x2b81)
  *
  * ATT: Read By Type Response (0x09) len 7
  *     Attribute data length: 6
  *     Attribute data list: 1 entry
  *     Handle: 0x0006
  *     Value: 02000000
  */
#define VOCS_READ_CHAR_AUD_LOC_UUID \
	IOV_DATA(0x08, 0x05, 0x00, 0x06, 0x00, 0x81, 0x2b), \
	IOV_DATA(0x09, 0x06, 0x06, 0x00, 0x02, 0x00, 0x00, 0x00)

 /* ATT: Write Request (0x12) len 6
  *     Handle: 0x0009
  *          Data: 01280a00
  *
  * ATT: Error Response (0x01) len 4
  *     Write Request (0x12)
  *     Handle: 0x0009
  *     Error: Reserved (0x80)
  */
#define VOCS_CP_INVALID_CHNG_COUNTER \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x28, 0x0a, 0x00),\
	IOV_DATA(0x01, 0x12, 0x09, 0x00, 0x80)

 /* ATT: Write Request (0x12) len 6
  *     Handle: 0x0009
  *         Data: 00007800
  *
  * ATT: Error Response (0x01) len 4
  *     Write Request (0x12)
  *     Handle: 0x0009
  *     Error: Reserved (0x81)
  */
#define VOCS_CP_INVALID_OPCODE \
	IOV_DATA(0x12, 0x09, 0x00, 0x00, 0x00, 0x78, 0x00),\
	IOV_DATA(0x01, 0x12, 0x09, 0x00, 0x81)

 /* ATT: Write Request (0x12) len 6
  *     Handle: 0x0009
  *         Data: 01000e01
  *
  * ATT: Error Response (0x01) len 4
  *     Write Request (0x12)
  *     Handle: 0x0009
  *     Error: Reserved (0x82)
  *
  * ATT: Write Request (0x12) len 6
  *     Handle: 0x0009
  *         Data: 0100f2fe
  *
  * ATT: Error Response (0x01) len 4
  *     Write Request (0x12)
  *     Handle: 0x0009
  *     Error: Reserved (0x82)
  */
#define VOCS_CP_OUT_OF_RANGE_VALUE \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x00, 0x0e, 0x01), \
	IOV_DATA(0x01, 0x12, 0x09, 0x00, 0x82), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x00, 0xf2, 0xfe),\
	IOV_DATA(0x01, 0x12, 0x09, 0x00, 0x82)

 /* ATT: Write Request (0x12) len 4
  *     Handle: 0x0004
  *         Data: 0100
  *
  * ATT: Write Response (0x13) len 0
  */
#define VOCS_ENNABLE_VOL_OFFSET_CCD \
	IOV_DATA(0x12, 0x04, 0x00, 0x01, 0x00), \
	IOV_DATA(0x13)

#define VOCS_SR_SGGIT_CHA_TST_CMDS \
	VOCS_EXCHANGE_MTU, \
	VOCS_PRIMARY_SERVICE_VCS, \
	VOCS_SECONDARY_SERVICE_VOCS, \
	VOCS_INCLUDED_SERVICE_VOCS, \
	VOCS_DISC_CHAR, \
	VOCS_DISC_CHAR_DESC, \
	VOCS_READ_CHAR_DESC

#define VOCS_SR_SGGIT_SER_BV_01_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS

#define VOCS_SR_SGGIT_CHA_BV_01_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS, \
	VOCS_READ_CHAR_VOL_OFFSET, \
	VOCS_READ_CHAR_VOL_OFFSET_UUID

#define VOCS_SR_SGGIT_CHA_BV_02_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS, \
	VOCS_READ_CHAR_AUD_LOC, \
	VOCS_READ_CHAR_AUD_LOC_UUID

#define VOCS_SR_SGGIT_CHA_BV_03_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS

#define VOCS_SR_SGGIT_CHA_BV_04_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS

#define VOCS_SR_SGGIT_CP_BI_01_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS, \
	VOCS_CP_INVALID_CHNG_COUNTER

#define VOCS_SR_SGGIT_CP_BI_02_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS, \
	VOCS_CP_INVALID_OPCODE

#define VOCS_SR_SGGIT_CP_BI_03_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS, \
	VOCS_CP_OUT_OF_RANGE_VALUE

#define VOCS_SR_SPE_BI_01_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS, \
	VOCS_READ_CHAR_AUD_LOC
 /*
  * VOCS/SR/CP/BV-01-C [Set Volume Offset]
  * Do Initial Condition Proedures
  * 1. The Lower Tester executes the GATT Read Characteristic
  *    Value sub-procedure for the Volume Offset State characteristic.
  * Repeat steps 2-4 for (255 - Change_Counter value) + 1 times.
  * 2. The Lower Tester executes the GATT Write Characteristic Value
  *    sub-procedure for the Volume
  *    Offset Control Point characteristic with the Set Volume Offset
  *    Opcode, with the Volume Offset parameters set to a random value
  *    different from the last one, and the Change_Counter parameter.
  * 3. The Lower Tester receives a Write Response indicating that the IUT
  *     has accepted the Opcode.
  * 4. The Lower Tester receives a GATT Characteristic Value Notification
  *    for the Volume Offset State characteristic.
  */
#define VOCS_SR_CP_BV_01_C \
	VOCS_SR_SGGIT_CHA_TST_CMDS, \
	VOCS_ENNABLE_VOL_OFFSET_CCD, \
	VOCS_READ_CHAR_VOL_OFFSET, \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x00, 0xde, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xde, 0x00, 0x01), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x01, 0xda, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xda, 0xff, 0x02), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x02, 0x1a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1a, 0x00, 0x03), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x03, 0x49, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x49, 0xff, 0x04), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x04, 0x05, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x05, 0xff, 0x05), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x05, 0xf1, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xf1, 0xff, 0x06), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x06, 0xca, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xca, 0xff, 0x07), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x07, 0x5c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x5c, 0x00, 0x08), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x08, 0xaf, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xaf, 0x00, 0x09), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x09, 0x5f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x5f, 0x00, 0x0a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x0a, 0x69, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x69, 0xff, 0x0b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x0b, 0x3d, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3d, 0xff, 0x0c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x0c, 0xb6, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xb6, 0xff, 0x0d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x0d, 0xa4, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa4, 0x00, 0x0e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x0e, 0x14, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x14, 0xff, 0x0f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x0f, 0x2a, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x2a, 0xff, 0x10), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x10, 0x51, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x51, 0x00, 0x11), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x011, 0xc4, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xc4, 0xff, 0x12), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x12, 0xe8, 00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xe8, 0x00, 0x13), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x13, 0xca, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xca, 0xff, 0x14), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x14, 0xe6, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xe6, 0xff, 0x15), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x15, 0x62, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x62, 0x00, 0x16), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x16, 0x22, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x22, 0x00, 0x17), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x17, 0xa1, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa1, 0xff, 0x18), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x18, 0xaa, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xaa, 0x00, 0x19), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x19, 0x65, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x65, 0x00, 0x1a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x1a, 0x11, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x11, 0xff, 0x1b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x1b, 0x69, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x69, 0xff, 0x1c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x1c, 0xee, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xee, 0x00, 0x1d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x1d, 0xaa, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xaa, 0xff, 0x1e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x1e, 0x1f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1f, 0xff, 0x1f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x1f, 0xbe, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xbe, 0x00, 0x20), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x20, 0x93, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x93, 0xff, 0x21), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x21, 0x11, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x11, 0xff, 0x22), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x22, 0x83, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x83, 0xff, 0x23), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x23, 0xf8, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xf8, 0x00, 0x24), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x24, 0x90, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x90, 0xff, 0x25), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x25, 0x0c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0c, 0x00, 0x26), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x26, 0xc8, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xc8, 0x00, 0x27), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x27, 0x59, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x59, 0xff, 0x28), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x28, 0x80, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x80, 0xff, 0x29), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x29, 0x0d, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0d, 0xff, 0x2a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x2a, 0x0c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0c, 0x00, 0x2b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x2b, 0x0a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0a, 0x00, 0x2c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x2c, 0x12, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x12, 0x00, 0x2d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x2d, 0x0b, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0b, 0xff, 0x2e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x2e, 0x83, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x83, 0xff, 0x2f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x2f, 0x91, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x91, 0xff, 0x30), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x30, 0x71, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x71, 0xff, 0x31), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x31, 0x72, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x72, 0xff, 0x32), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x32, 0x75, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x75, 0xff, 0x33), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x33, 0x78, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x78, 0xff, 0x34), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x34, 0x61, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x61, 0xff, 0x35), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x35, 0x63, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x63, 0xff, 0x36), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x36, 0x38, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x38, 0xff, 0x37), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x37, 0x21, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x21, 0xff, 0x38), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x38, 0xa4, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa4, 0x00, 0x39), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x39, 0xb4, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xb4, 0x00, 0x3a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x3a, 0xb5, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xb5, 0xff, 0x3b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x3b, 0xac, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xac, 0x00, 0x3c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x3c, 0xab, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xab, 0x00, 0x3d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x3d, 0xad, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xad, 0x00, 0x3e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x3e, 0x83, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x83, 0xff, 0x3f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x3f, 0x84, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x84, 0xff, 0x40), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x40, 0x85, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x85, 0xff, 0x41), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x41, 0x86, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x86, 0xff, 0x42), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x42, 0x87, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x87, 0xff, 0x43), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x43, 0x87, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x87, 0xff, 0x44), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x44, 0x05, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x05, 0x000, 0x45), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x45, 0xce, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xce, 0x00, 0x46), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x46, 0x96, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x96, 0x00, 0x47), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x47, 0x07, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x07, 0x00, 0x48), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x48, 0x08, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x08, 0x00, 0x49), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x49, 0x09, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x09, 0xff, 0x4a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x4a, 0x0a, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0a, 0xff, 0x4b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x4b, 0x11, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x11, 0xff, 0x4c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x4c, 0x22, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x22, 0xff, 0x4d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x4d, 0x33, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x33, 0xff, 0x4e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x4e, 0x09, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x09, 0xff, 0x4f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x4f, 0x19, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x19, 0xff, 0x50), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x50, 0x1a, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1a, 0xff, 0x51), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x51, 0x1b, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1b, 0xff, 0x52), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x52, 0xa1, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa1, 0x00, 0x53), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x53, 0xa2, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa2, 0x00, 0x54), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x54, 0xb2, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xb2, 0x00, 0x55), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x55, 0xb3, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xb3, 0x00, 0x56), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x56, 0x68, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x68, 0x00, 0x57), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x57, 0x69, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x69, 0x00, 0x58), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x58, 0x6a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x6a, 0x00, 0x59), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x59, 0x7a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7a, 0x00, 0x5a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x5a, 0x7b, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7b, 0x00, 0x5b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x5b, 0x8c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x8c, 0x00, 0x5c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x5c, 0x9c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9c, 0x00, 0x5d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x5d, 0x9b, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9b, 0x00, 0x5e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x5e, 0x9c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9c, 0x00, 0x5f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x5f, 0x9d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9d, 0x00, 0x60), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x60, 0x9e, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9e, 0x00, 0x61), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x61, 0x21, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x21, 0x00, 0x62), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x62, 0x23, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x23, 0x00, 0x63), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x63, 0x24, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x24, 0x00, 0x64), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x64, 0x34, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x34, 0x00, 0x65), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x65, 0x44, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x44, 0x00, 0x66), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x66, 0x45, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x45, 0x00, 0x67), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x67, 0x9d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9d, 0x00, 0x68), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x68, 0x9d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9d, 0x00, 0x69), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x69, 0x9d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9d, 0x00, 0x6a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x6a, 0x49, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x49, 0x00, 0x6b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x6b, 0x39, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x39, 0x00, 0x6c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x6c, 0x9d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9d, 0x00, 0x6d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x6d, 0x9e, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9e, 0x00, 0x6e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x6e, 0x9f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9f, 0x00, 0x6f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x6f, 0x91, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x91, 0x00, 0x70), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x70, 0x18, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x18, 0x00, 0x71), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x71, 0x34, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x34, 0xff, 0x72), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x72, 0x44, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x44, 0xff, 0x73), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x73, 0x05, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x05, 0xff, 0x74), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x74, 0x06, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x06, 0xff, 0x75), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x75, 0x38, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x38, 0x00, 0x76), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x76, 0x48, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x48, 0x00, 0x77), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x77, 0x58, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x58, 0x00, 0x78), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x78, 0x88, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x88, 0x00, 0x79), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x79, 0x98, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x98, 0x00, 0x7a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x7a, 0x91, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x91, 0x00, 0x7b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x7b, 0x95, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x95, 0x00, 0x7c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x7c, 0x89, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x89, 0x00, 0x7d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x7d, 0x82, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x82, 0x00, 0x7e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x7e, 0x88, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x88, 0x00, 0x7f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x7f, 0x66, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x66, 0x00, 0x80), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x80, 0x55, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x55, 0x00, 0x81), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x81, 0x44, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x44, 0x00, 0x82), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x82, 0x33, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x33, 0x00, 0x83), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x83, 0x22, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x22, 0x00, 0x84), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x84, 0x11, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x11, 0x00, 0x85), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x85, 0x01, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x01, 0x00, 0x86), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x86, 0x3a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3a, 0x00, 0x87), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x87, 0x3b, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3b, 0x00, 0x88), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x88, 0x3c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3c, 0x00, 0x89), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x89, 0x4c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x4c, 0x00, 0x8a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x8a, 0x5c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x5c, 0x00, 0x8b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x8b, 0x6c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x6c, 0x00, 0x8c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x8c, 0xab, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xab, 0xff, 0x8d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x8d, 0xac, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xac, 0xff, 0x8e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x8e, 0xbc, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xbc, 0x00, 0x8f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x8f, 0xbb, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xbb, 0x00, 0x90), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x90, 0x11, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x11, 0x00, 0x91), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x91, 0x21, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x21, 0x00, 0x92), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x92, 0x31, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x31, 0x00, 0x93), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x93, 0x21, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x21, 0x00, 0x94), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x94, 0x31, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x31, 0x00, 0x95), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x95, 0x41, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x41, 0x00, 0x96), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x96, 0x51, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x51, 0x00, 0x97), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x97, 0x61, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x61, 0x00, 0x98), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x98, 0x81, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x81, 0x00, 0x99), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x99, 0x55, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x55, 0x00, 0x9a), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x9a, 0x59, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x59, 0x00, 0x9b), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x9b, 0x56, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x56, 0x00, 0x9c), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x9c, 0x57, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x57, 0x00, 0x9d), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x9d, 0x58, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x58, 0x00, 0x9e), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x9e, 0x59, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x59, 0x00, 0x9f), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0x9f, 0x60, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x60, 0x00, 0xa0), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa0, 0x0b, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0b, 0xff, 0xa1), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa1, 0x0c, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0c, 0xff, 0xa2), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa2, 0x0c, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0c, 0xff, 0xa3), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa3, 0x0d, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0d, 0xff, 0xa4), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa4, 0x53, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x53, 0xff, 0xa5), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa5, 0x54, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x54, 0xff, 0xa6), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa6, 0x75, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x75, 0xff, 0xa7), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa7, 0x76, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x76, 0xff, 0xa8), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa8, 0x77, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x77, 0xff, 0xa9), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xa9, 0x78, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x78, 0xff, 0xaa), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xaa, 0x76, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x76, 0xff, 0xab), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xab, 0xa1, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa1, 0x00, 0xac), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xac, 0xc1, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xc1, 0x00, 0xad), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xad, 0xd1, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xd1, 0x00, 0xae), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xae, 0xe1, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xe1, 0x00, 0xaf), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xaf, 0xf1, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xf1, 0x00, 0xb0), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb0, 0xae, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xae, 0x00, 0xb1), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb1, 0xbe, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xbe, 0x00, 0xb2), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb2, 0xdd, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xdd, 0x00, 0xb3), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb3, 0xee, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xee, 0x00, 0xb4), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb4, 0x1d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1d, 0x00, 0xb5), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb5, 0x3a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3a, 0x00, 0xb6), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb6, 0x4a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x4a, 0x00, 0xb7), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb7, 0x5a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x5a, 0x00, 0xb8), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb8, 0x7e, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7e, 0x00, 0xb9), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xb9, 0x3f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3f, 0x00, 0xba), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xba, 0x3f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3f, 0x00, 0xbb), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xbb, 0xa1, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa1, 0x00, 0xbc), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xbc, 0xa2, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa2, 0x00, 0xbd), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xbd, 0xa3, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa3, 0x00, 0xbe), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xbe, 0xa4, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa4, 0x00, 0xbf), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xbf, 0xa5, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa5, 0x00, 0xc0), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc0, 0xa6, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xa6, 0x00, 0xc1), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc1, 0x1f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1f, 0x00, 0xc2), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc2, 0x2f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x2f, 0x00, 0xc3), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc3, 0x3f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3f, 0x00, 0xc4), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc4, 0x4f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x4f, 0x00, 0xc5), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc5, 0x5f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x5f, 0x00, 0xc6), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc6, 0x6f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x6f, 0x00, 0xc7), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc7, 0x7f, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7f, 0x00, 0xc8), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc8, 0x1d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1d, 0x00, 0xc9), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xc9, 0xaa, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xaa, 0x00, 0xca), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xca, 0xbb, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xbb, 0x00, 0xcb), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xcb, 0xcd, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xcd, 0x00, 0xcc), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xcc, 0xce, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xce, 0x00, 0xcd), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xcd, 0xde, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xde, 0x00, 0xce), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xce, 0xdf, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xdf, 0x00, 0xcf), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xcf, 0xdb, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xdb, 0x00, 0xd0), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd0, 0x6e, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x6e, 0x00, 0xd1), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd1, 0x5e, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x5e, 0x00, 0xd2), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd2, 0x8e, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x8e, 0x00, 0xd3), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd3, 0x9e, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x9e, 0x00, 0xd4), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd4, 0xae, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xae, 0x00, 0xd5), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd5, 0xbe, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xbe, 0x00, 0xd6), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd6, 0xee, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xee, 0x00, 0xd7), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd7, 0x1c, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1c, 0x00, 0xd8), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd8, 0x33, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x33, 0x00, 0xd9), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xd9, 0x88, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x88, 0x00, 0xda), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xda, 0x0d, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0d, 0x00, 0xdb), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xdb, 0x88, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x88, 0x00, 0xdc), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xdc, 0x99, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x99, 0x00, 0xdd), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xdd, 0x66, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x66, 0x00, 0xde), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xde, 0x49, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x49, 0x00, 0xdf), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xdf, 0x86, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x86, 0x00, 0xe0), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe0, 0x3a, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3a, 0x00, 0xe1), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe1, 0xd0, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xd0, 0x00, 0xe2), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe2, 0xd2, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xd2, 0x00, 0xe3), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe3, 0xd3, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xd3, 0x00, 0xe4), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe4, 0xd4, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xd4, 0x00, 0xe5), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe5, 0xdf, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xdf, 0x00, 0xe6), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe6, 0xef, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xef, 0x00, 0xe7), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe7, 0xed, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xed, 0x00, 0xe8), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe8, 0xcc, 0x00), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xcc, 0x00, 0xe9), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xe9, 0x1f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x1f, 0xff, 0xea), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xea, 0x2f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x2f, 0xff, 0xeb), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xeb, 0x3f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x3f, 0xff, 0xec), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xec, 0x4f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x4f, 0xff, 0xed), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xed, 0x5f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x5f, 0xff, 0xee), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xee, 0x6f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x6f, 0xff, 0xef), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xef, 0x7f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7f, 0xff, 0xf0), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf0, 0xc9, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xc9, 0xff, 0xf1), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf1, 0xb9, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xb9, 0xff, 0xf2), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf2, 0xd9, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xd9, 0xff, 0xf3), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf3, 0xe1, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0xe1, 0xff, 0xf4), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf4, 0x8f, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x8f, 0xff, 0xf5), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf5, 0x7a, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7a, 0xff, 0xf6), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf6, 0x7b, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7b, 0xff, 0xf7), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf7, 0x7c, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7c, 0xff, 0xf8), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf8, 0x7d, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7d, 0xff, 0xf9), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xf9, 0x7e, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7e, 0xff, 0xfa), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xfa, 0x6a, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x6a, 0xff, 0xfb), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xfb, 0x6b, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x6b, 0xff, 0xfc), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xfc, 0x7e, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x7e, 0xff, 0xfd), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xfd, 0x0a, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0a, 0xff, 0xfe), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xfe, 0x0b, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0b, 0xff, 0xff), \
	IOV_DATA(0x12, 0x09, 0x00, 0x01, 0xff, 0x0c, 0xff), \
	IOV_DATA(0x13), \
	IOV_DATA(0x1b, 0x03, 0x00, 0x0c, 0xff, 0x00)

int main(int argc, char *argv[])
{
	tester_init(&argc, &argv);

	/* VOCS Unit Testcases */
	define_test("VOCS/SR/SGGIT/SER/BV-01-C", test_server, NULL,
					VOCS_SR_SGGIT_SER_BV_01_C);

	define_test("VOCS/SR/SGGIT/CHA/BV-01-C", test_server, NULL,
					VOCS_SR_SGGIT_CHA_BV_01_C);
	define_test("VOCS/SR/SGGIT/CHA/BV-02-C", test_server, NULL,
					VOCS_SR_SGGIT_CHA_BV_02_C);
	define_test("VOCS/SR/SGGIT/CHA/BV-03-C", test_server, NULL,
					VOCS_SR_SGGIT_CHA_BV_03_C);
	define_test("VOCS/SR/SGGIT/CHA/BV-04-C", test_server, NULL,
					VOCS_SR_SGGIT_CHA_BV_04_C);

	define_test("VOCS/SR/SGGIT/CP/BI-01-C", test_server, NULL,
					VOCS_SR_SGGIT_CP_BI_01_C);
	define_test("VOCS/SR/SGGIT/CP/BI-02-C", test_server, NULL,
					VOCS_SR_SGGIT_CP_BI_02_C);
	define_test("VOCS/SR/SGGIT/CP/BI-03-C", test_server, NULL,
					VOCS_SR_SGGIT_CP_BI_03_C);

	define_test("VOCS/SR/SPE/BI-01-C", test_server, NULL,
					VOCS_SR_SPE_BI_01_C);

	define_test("VOCS/SR/CP/BV-01-C", test_server, NULL,
					VOCS_SR_CP_BV_01_C);

	return tester_run();
}

