/*
 * Copyright (c) 2016, ARM Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <arm_gic.h>
#include <assert.h>
#include <bl_common.h>
#include <console.h>
#include <debug.h>
#include <generic_delay_timer.h>
#include <mmio.h>
#include <platform.h>
#include <plat_private.h>
#include <platform_def.h>

/*******************************************************************************
 * Declarations of linker defined symbols which will help us find the layout
 * of trusted SRAM
 ******************************************************************************/
unsigned long __RO_START__;
unsigned long __RO_END__;

/*
 * The next 2 constants identify the extents of the code & RO data region.
 * These addresses are used by the MMU setup code and therefore they must be
 * page-aligned.  It is the responsibility of the linker script to ensure that
 * __RO_START__ and __RO_END__ linker symbols refer to page-aligned addresses.
 */
#define BL31_RO_BASE (unsigned long)(&__RO_START__)
#define BL31_RO_LIMIT (unsigned long)(&__RO_END__)

#define BL31_TEXT_START (unsigned long)(&__TEXT_START__)
#define BL31_RO_END (unsigned long)(&__RODATA_END__)

#define BL31_END (uintptr_t)(&__BL31_END__)

static entry_point_info_t bl32_ep_info;
static entry_point_info_t bl33_ep_info;

/*******************************************************************************
 * Return a pointer to the 'entry_point_info' structure of the next image for
 * the security state specified. BL33 corresponds to the non-secure image type
 * while BL32 corresponds to the secure image type. A NULL pointer is returned
 * if the image does not exist.
 ******************************************************************************/
entry_point_info_t *bl31_plat_get_next_image_ep_info(uint32_t type)
{
#if 0
        assert(sec_state_is_valid(type));
        assert(type == NON_SECURE);

        return &bl33_ep_info;
#else
	entry_point_info_t *next_image_info;

	next_image_info = (type == NON_SECURE) ? &bl33_ep_info : &bl32_ep_info;

	/* None of the images on this platform can have 0x0 as the entrypoint */
	if (next_image_info->pc)
		return next_image_info;
	else
		return NULL;
#endif
}

/*******************************************************************************
 * Perform any BL3-1 early platform setup. Here is an opportunity to copy
 * parameters passed by the calling EL (S-EL1 in BL2 & S-EL3 in BL1) before they
 * are lost (potentially). This needs to be done before the MMU is initialized
 * so that the memory layout can be used while creating page tables.
 * BL2 has flushed this information to memory, so we are guaranteed to pick up
 * good data.
 ******************************************************************************/
#include <arch.h>
#include <arch_helpers.h>

void bl31_early_platform_setup(bl31_params_t *from_bl2,
			       void *plat_params_from_bl2)
{
	console_init(PLAT_RK_UART_BASE, PLAT_RK_UART_CLOCK,
		     PLAT_RK_UART_BAUDRATE);

	INFO("%s: from_bl2 %p plat_params_from_bl2 %p",
	     __func__, (void*)from_bl2, plat_params_from_bl2);

	/* Passing a NULL context is a critical programming error */
	assert(from_bl2);

	INFO("%s: from_bl2->h.type %x\n",
	     __func__, from_bl2->h.type);
	assert(from_bl2->h.type == PARAM_BL31);
	INFO("%s: from_bl2->h.version %x\n",
	     __func__, from_bl2->h.version);
	assert(from_bl2->h.version >= VERSION_1);

	bl32_ep_info = *from_bl2->bl32_ep_info;
	bl33_ep_info = *from_bl2->bl33_ep_info;

	/* there may have some board specific data */
	params_early_setup(plat_params_from_bl2);

	INFO("bl33_ep_info.pc: %p\n", (void*)(bl33_ep_info.pc));
	plat_rockchip_pmusram_prepare();
}

/*******************************************************************************
 * Perform any BL3-1 platform setup code
 ******************************************************************************/
void bl31_platform_setup(void)
{
	generic_delay_timer_init();
	plat_rockchip_soc_init();

	/* Initialize the gic cpu and distributor interfaces */
	plat_rockchip_gic_driver_init();
	plat_rockchip_gic_init();
	plat_rockchip_pmu_init();
}

/*******************************************************************************
 * Perform the very early platform specific architectural setup here. At the
 * moment this is only intializes the mmu in a quick and dirty way.
 ******************************************************************************/
/*
 * Set up the page tables for the generic and platform-specific memory regions.
 * The extents of the generic memory regions are specified by the function
 * arguments and consist of:
 * - Trusted SRAM seen by the BL image;
 * - Code section;
 * - Read-only data section;
 * - Coherent memory region, if applicable.
 */

#if 0
const mmap_region_t plat_arm_mmap[] = {
  { UART0_BASE, UART0_BASE, 0x10000, MT_DEVICE | MT_RW | MT_SECURE },
  { 0 }
};
#endif

const mmap_region_t *plat_arm_get_mmap(void)
{
	return plat_rk_mmap;
}

void arm_setup_page_tables(uintptr_t total_base,
			   size_t total_size,
			   uintptr_t code_start,
			   uintptr_t code_limit,
			   uintptr_t rodata_start,
			   uintptr_t rodata_limit
#if USE_COHERENT_MEM
			   ,
			   uintptr_t coh_start,
			   uintptr_t coh_limit
#endif
			   )
{
	/*
	 * Map the Trusted SRAM with appropriate memory attributes.
	 * Subsequent mappings will adjust the attributes for specific regions.
	 */
	VERBOSE("Trusted SRAM seen by this BL image: %p - %p\n",
		(void *) total_base, (void *) (total_base + total_size));
	mmap_add_region(total_base, total_base,
			total_size,
			MT_MEMORY | MT_RW | MT_SECURE);

	/* Re-map the code section */
	VERBOSE("Code region: %p - %p\n",
		(void *) code_start, (void *) code_limit);
	mmap_add_region(code_start, code_start,
			code_limit - code_start,
			MT_CODE | MT_SECURE);

	/* Re-map the read-only data section */
	VERBOSE("Read-only data region: %p - %p\n",
		(void *) rodata_start, (void *) rodata_limit);
	mmap_add_region(rodata_start, rodata_start,
			rodata_limit - rodata_start,
			MT_RO_DATA | MT_SECURE);

#if USE_COHERENT_MEM
	/* Re-map the coherent memory region */
	VERBOSE("Coherent region: %p - %p\n",
		(void *) coh_start, (void *) coh_limit);
	mmap_add_region(coh_start, coh_start,
			coh_limit - coh_start,
			MT_DEVICE | MT_RW | MT_SECURE);
#endif

	/* Now (re-)map the platform-specific memory regions */
	mmap_add(plat_arm_get_mmap());

	/* Create the page tables to reflect the above mappings */
	init_xlat_tables();
}

void bl31_plat_arch_setup(void)
{
	plat_cci_init();
	plat_cci_enable();

	tf_printf("BL31_RO_BASE: %lx\n", BL31_RO_BASE);
	tf_printf("BL31_RO_LIMIT: %lx\n", BL31_RO_LIMIT);
	tf_printf("BL_COHERENT_RAM_BASE: %lx\n", BL_COHERENT_RAM_BASE);
	tf_printf("BL_COHERENT_RAM_END: %lx\n", BL_COHERENT_RAM_END);

#if 0
	plat_configure_mmu_el3(BL31_RO_BASE,
			       BL_COHERENT_RAM_END - BL31_RO_BASE,
			       BL31_RO_BASE,
			       BL31_RO_LIMIT,
			       BL_COHERENT_RAM_BASE,
			       BL_COHERENT_RAM_END);
#else
	arm_setup_page_tables(BL31_BASE,
			      BL31_END - BL31_BASE,
			      BL_CODE_BASE,
			      BL_CODE_END,
			      BL_RO_DATA_BASE,
			      BL_RO_DATA_END
#if USE_COHERENT_MEM
			      , BL_COHERENT_RAM_BASE,
			      BL_COHERENT_RAM_END
#endif
			      );
	enable_mmu_el3(0);
#endif

	tf_printf("%s: still alive\n", __func__);
}
