/* $OpenBSD$ */

/*
 * Copyright (c) 2008 Nicholas Marriott <nicm@users.sourceforge.net>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "tmux.h"

/*
 * Colour to string conversion functions. Bit 8 of the colour means it is one
 * of the 256 colour palette.
 */

struct colour_rgb {
	u_char  i;
	u_char	r;
	u_char	g;
	u_char	b;
};

const struct colour_rgb colour_from_256[] = {
	{   0, 0x00, 0x00, 0x00 }, {   1, 0x00, 0x00, 0x5f },
	{   2, 0x00, 0x00, 0x87 }, {   3, 0x00, 0x00, 0xaf },
	{   4, 0x00, 0x00, 0xd7 }, {   5, 0x00, 0x00, 0xff },
	{   6, 0x00, 0x5f, 0x00 }, {   7, 0x00, 0x5f, 0x5f },
	{   8, 0x00, 0x5f, 0x87 }, {   9, 0x00, 0x5f, 0xaf },
	{  10, 0x00, 0x5f, 0xd7 }, {  11, 0x00, 0x5f, 0xff },
	{  12, 0x00, 0x87, 0x00 }, {  13, 0x00, 0x87, 0x5f },
	{  14, 0x00, 0x87, 0x87 }, {  15, 0x00, 0x87, 0xaf },
	{  16, 0x00, 0x87, 0xd7 }, {  17, 0x00, 0x87, 0xff },
	{  18, 0x00, 0xaf, 0x00 }, {  19, 0x00, 0xaf, 0x5f },
	{  20, 0x00, 0xaf, 0x87 }, {  21, 0x00, 0xaf, 0xaf },
	{  22, 0x00, 0xaf, 0xd7 }, {  23, 0x00, 0xaf, 0xff },
	{  24, 0x00, 0xd7, 0x00 }, {  25, 0x00, 0xd7, 0x5f },
	{  26, 0x00, 0xd7, 0x87 }, {  27, 0x00, 0xd7, 0xaf },
	{  28, 0x00, 0xd7, 0xd7 }, {  29, 0x00, 0xd7, 0xff },
	{  30, 0x00, 0xff, 0x00 }, {  31, 0x00, 0xff, 0x5f },
	{  32, 0x00, 0xff, 0x87 }, {  33, 0x00, 0xff, 0xaf },
	{  34, 0x00, 0xff, 0xd7 }, {  35, 0x00, 0xff, 0xff },
	{  36, 0x5f, 0x00, 0x00 }, {  37, 0x5f, 0x00, 0x5f },
	{  38, 0x5f, 0x00, 0x87 }, {  39, 0x5f, 0x00, 0xaf },
	{  40, 0x5f, 0x00, 0xd7 }, {  41, 0x5f, 0x00, 0xff },
	{  42, 0x5f, 0x5f, 0x00 }, {  43, 0x5f, 0x5f, 0x5f },
	{  44, 0x5f, 0x5f, 0x87 }, {  45, 0x5f, 0x5f, 0xaf },
	{  46, 0x5f, 0x5f, 0xd7 }, {  47, 0x5f, 0x5f, 0xff },
	{  48, 0x5f, 0x87, 0x00 }, {  49, 0x5f, 0x87, 0x5f },
	{  50, 0x5f, 0x87, 0x87 }, {  51, 0x5f, 0x87, 0xaf },
	{  52, 0x5f, 0x87, 0xd7 }, {  53, 0x5f, 0x87, 0xff },
	{  54, 0x5f, 0xaf, 0x00 }, {  55, 0x5f, 0xaf, 0x5f },
	{  56, 0x5f, 0xaf, 0x87 }, {  57, 0x5f, 0xaf, 0xaf },
	{  58, 0x5f, 0xaf, 0xd7 }, {  59, 0x5f, 0xaf, 0xff },
	{  60, 0x5f, 0xd7, 0x00 }, {  61, 0x5f, 0xd7, 0x5f },
	{  62, 0x5f, 0xd7, 0x87 }, {  63, 0x5f, 0xd7, 0xaf },
	{  64, 0x5f, 0xd7, 0xd7 }, {  65, 0x5f, 0xd7, 0xff },
	{  66, 0x5f, 0xff, 0x00 }, {  67, 0x5f, 0xff, 0x5f },
	{  68, 0x5f, 0xff, 0x87 }, {  69, 0x5f, 0xff, 0xaf },
	{  70, 0x5f, 0xff, 0xd7 }, {  71, 0x5f, 0xff, 0xff },
	{  72, 0x87, 0x00, 0x00 }, {  73, 0x87, 0x00, 0x5f },
	{  74, 0x87, 0x00, 0x87 }, {  75, 0x87, 0x00, 0xaf },
	{  76, 0x87, 0x00, 0xd7 }, {  77, 0x87, 0x00, 0xff },
	{  78, 0x87, 0x5f, 0x00 }, {  79, 0x87, 0x5f, 0x5f },
	{  80, 0x87, 0x5f, 0x87 }, {  81, 0x87, 0x5f, 0xaf },
	{  82, 0x87, 0x5f, 0xd7 }, {  83, 0x87, 0x5f, 0xff },
	{  84, 0x87, 0x87, 0x00 }, {  85, 0x87, 0x87, 0x5f },
	{  86, 0x87, 0x87, 0x87 }, {  87, 0x87, 0x87, 0xaf },
	{  88, 0x87, 0x87, 0xd7 }, {  89, 0x87, 0x87, 0xff },
	{  90, 0x87, 0xaf, 0x00 }, {  91, 0x87, 0xaf, 0x5f },
	{  92, 0x87, 0xaf, 0x87 }, {  93, 0x87, 0xaf, 0xaf },
	{  94, 0x87, 0xaf, 0xd7 }, {  95, 0x87, 0xaf, 0xff },
	{  96, 0x87, 0xd7, 0x00 }, {  97, 0x87, 0xd7, 0x5f },
	{  98, 0x87, 0xd7, 0x87 }, {  99, 0x87, 0xd7, 0xaf },
	{ 100, 0x87, 0xd7, 0xd7 }, { 101, 0x87, 0xd7, 0xff },
	{ 102, 0x87, 0xff, 0x00 }, { 103, 0x87, 0xff, 0x5f },
	{ 104, 0x87, 0xff, 0x87 }, { 105, 0x87, 0xff, 0xaf },
	{ 106, 0x87, 0xff, 0xd7 }, { 107, 0x87, 0xff, 0xff },
	{ 108, 0xaf, 0x00, 0x00 }, { 109, 0xaf, 0x00, 0x5f },
	{ 110, 0xaf, 0x00, 0x87 }, { 111, 0xaf, 0x00, 0xaf },
	{ 112, 0xaf, 0x00, 0xd7 }, { 113, 0xaf, 0x00, 0xff },
	{ 114, 0xaf, 0x5f, 0x00 }, { 115, 0xaf, 0x5f, 0x5f },
	{ 116, 0xaf, 0x5f, 0x87 }, { 117, 0xaf, 0x5f, 0xaf },
	{ 118, 0xaf, 0x5f, 0xd7 }, { 119, 0xaf, 0x5f, 0xff },
	{ 120, 0xaf, 0x87, 0x00 }, { 121, 0xaf, 0x87, 0x5f },
	{ 122, 0xaf, 0x87, 0x87 }, { 123, 0xaf, 0x87, 0xaf },
	{ 124, 0xaf, 0x87, 0xd7 }, { 125, 0xaf, 0x87, 0xff },
	{ 126, 0xaf, 0xaf, 0x00 }, { 127, 0xaf, 0xaf, 0x5f },
	{ 128, 0xaf, 0xaf, 0x87 }, { 129, 0xaf, 0xaf, 0xaf },
	{ 130, 0xaf, 0xaf, 0xd7 }, { 131, 0xaf, 0xaf, 0xff },
	{ 132, 0xaf, 0xd7, 0x00 }, { 133, 0xaf, 0xd7, 0x5f },
	{ 134, 0xaf, 0xd7, 0x87 }, { 135, 0xaf, 0xd7, 0xaf },
	{ 136, 0xaf, 0xd7, 0xd7 }, { 137, 0xaf, 0xd7, 0xff },
	{ 138, 0xaf, 0xff, 0x00 }, { 139, 0xaf, 0xff, 0x5f },
	{ 140, 0xaf, 0xff, 0x87 }, { 141, 0xaf, 0xff, 0xaf },
	{ 142, 0xaf, 0xff, 0xd7 }, { 143, 0xaf, 0xff, 0xff },
	{ 144, 0xd7, 0x00, 0x00 }, { 145, 0xd7, 0x00, 0x5f },
	{ 146, 0xd7, 0x00, 0x87 }, { 147, 0xd7, 0x00, 0xaf },
	{ 148, 0xd7, 0x00, 0xd7 }, { 149, 0xd7, 0x00, 0xff },
	{ 150, 0xd7, 0x5f, 0x00 }, { 151, 0xd7, 0x5f, 0x5f },
	{ 152, 0xd7, 0x5f, 0x87 }, { 153, 0xd7, 0x5f, 0xaf },
	{ 154, 0xd7, 0x5f, 0xd7 }, { 155, 0xd7, 0x5f, 0xff },
	{ 156, 0xd7, 0x87, 0x00 }, { 157, 0xd7, 0x87, 0x5f },
	{ 158, 0xd7, 0x87, 0x87 }, { 159, 0xd7, 0x87, 0xaf },
	{ 160, 0xd7, 0x87, 0xd7 }, { 161, 0xd7, 0x87, 0xff },
	{ 162, 0xd7, 0xaf, 0x00 }, { 163, 0xd7, 0xaf, 0x5f },
	{ 164, 0xd7, 0xaf, 0x87 }, { 165, 0xd7, 0xaf, 0xaf },
	{ 166, 0xd7, 0xaf, 0xd7 }, { 167, 0xd7, 0xaf, 0xff },
	{ 168, 0xd7, 0xd7, 0x00 }, { 169, 0xd7, 0xd7, 0x5f },
	{ 170, 0xd7, 0xd7, 0x87 }, { 171, 0xd7, 0xd7, 0xaf },
	{ 172, 0xd7, 0xd7, 0xd7 }, { 173, 0xd7, 0xd7, 0xff },
	{ 174, 0xd7, 0xff, 0x00 }, { 175, 0xd7, 0xff, 0x5f },
	{ 176, 0xd7, 0xff, 0x87 }, { 177, 0xd7, 0xff, 0xaf },
	{ 178, 0xd7, 0xff, 0xd7 }, { 179, 0xd7, 0xff, 0xff },
	{ 180, 0xff, 0x00, 0x00 }, { 181, 0xff, 0x00, 0x5f },
	{ 182, 0xff, 0x00, 0x87 }, { 183, 0xff, 0x00, 0xaf },
	{ 184, 0xff, 0x00, 0xd7 }, { 185, 0xff, 0x00, 0xff },
	{ 186, 0xff, 0x5f, 0x00 }, { 187, 0xff, 0x5f, 0x5f },
	{ 188, 0xff, 0x5f, 0x87 }, { 189, 0xff, 0x5f, 0xaf },
	{ 190, 0xff, 0x5f, 0xd7 }, { 191, 0xff, 0x5f, 0xff },
	{ 192, 0xff, 0x87, 0x00 }, { 193, 0xff, 0x87, 0x5f },
	{ 194, 0xff, 0x87, 0x87 }, { 195, 0xff, 0x87, 0xaf },
	{ 196, 0xff, 0x87, 0xd7 }, { 197, 0xff, 0x87, 0xff },
	{ 198, 0xff, 0xaf, 0x00 }, { 199, 0xff, 0xaf, 0x5f },
	{ 200, 0xff, 0xaf, 0x87 }, { 201, 0xff, 0xaf, 0xaf },
	{ 202, 0xff, 0xaf, 0xd7 }, { 203, 0xff, 0xaf, 0xff },
	{ 204, 0xff, 0xd7, 0x00 }, { 205, 0xff, 0xd7, 0x5f },
	{ 206, 0xff, 0xd7, 0x87 }, { 207, 0xff, 0xd7, 0xaf },
	{ 208, 0xff, 0xd7, 0xd7 }, { 209, 0xff, 0xd7, 0xff },
	{ 210, 0xff, 0xff, 0x00 }, { 211, 0xff, 0xff, 0x5f },
	{ 212, 0xff, 0xff, 0x87 }, { 213, 0xff, 0xff, 0xaf },
	{ 214, 0xff, 0xff, 0xd7 }, { 215, 0xff, 0xff, 0xff },
	{ 216, 0x08, 0x08, 0x08 }, { 217, 0x12, 0x12, 0x12 },
	{ 218, 0x1c, 0x1c, 0x1c }, { 219, 0x26, 0x26, 0x26 },
	{ 220, 0x30, 0x30, 0x30 }, { 221, 0x3a, 0x3a, 0x3a },
	{ 222, 0x44, 0x44, 0x44 }, { 223, 0x4e, 0x4e, 0x4e },
	{ 224, 0x58, 0x58, 0x58 }, { 225, 0x62, 0x62, 0x62 },
	{ 226, 0x6c, 0x6c, 0x6c }, { 227, 0x76, 0x76, 0x76 },
	{ 228, 0x80, 0x80, 0x80 }, { 229, 0x8a, 0x8a, 0x8a },
	{ 230, 0x94, 0x94, 0x94 }, { 231, 0x9e, 0x9e, 0x9e },
	{ 232, 0xa8, 0xa8, 0xa8 }, { 233, 0xb2, 0xb2, 0xb2 },
	{ 234, 0xbc, 0xbc, 0xbc }, { 235, 0xc6, 0xc6, 0xc6 },
	{ 236, 0xd0, 0xd0, 0xd0 }, { 237, 0xda, 0xda, 0xda },
	{ 238, 0xe4, 0xe4, 0xe4 }, { 239, 0xee, 0xee, 0xee },
};
const struct colour_rgb colour_to_256[] = {
	{   0, 0x00, 0x00, 0x00 }, {   1, 0x00, 0x00, 0x5f },
	{   2, 0x00, 0x00, 0x87 }, {   3, 0x00, 0x00, 0xaf },
	{   4, 0x00, 0x00, 0xd7 }, {   5, 0x00, 0x00, 0xff },
	{   6, 0x00, 0x5f, 0x00 }, {   7, 0x00, 0x5f, 0x5f },
	{   8, 0x00, 0x5f, 0x87 }, {   9, 0x00, 0x5f, 0xaf },
	{  10, 0x00, 0x5f, 0xd7 }, {  11, 0x00, 0x5f, 0xff },
	{  12, 0x00, 0x87, 0x00 }, {  13, 0x00, 0x87, 0x5f },
	{  14, 0x00, 0x87, 0x87 }, {  15, 0x00, 0x87, 0xaf },
	{  16, 0x00, 0x87, 0xd7 }, {  17, 0x00, 0x87, 0xff },
	{  18, 0x00, 0xaf, 0x00 }, {  19, 0x00, 0xaf, 0x5f },
	{  20, 0x00, 0xaf, 0x87 }, {  21, 0x00, 0xaf, 0xaf },
	{  22, 0x00, 0xaf, 0xd7 }, {  23, 0x00, 0xaf, 0xff },
	{  24, 0x00, 0xd7, 0x00 }, {  25, 0x00, 0xd7, 0x5f },
	{  26, 0x00, 0xd7, 0x87 }, {  27, 0x00, 0xd7, 0xaf },
	{  28, 0x00, 0xd7, 0xd7 }, {  29, 0x00, 0xd7, 0xff },
	{  30, 0x00, 0xff, 0x00 }, {  31, 0x00, 0xff, 0x5f },
	{  32, 0x00, 0xff, 0x87 }, {  33, 0x00, 0xff, 0xaf },
	{  34, 0x00, 0xff, 0xd7 }, {  35, 0x00, 0xff, 0xff },
	{ 216, 0x08, 0x08, 0x08 }, { 217, 0x12, 0x12, 0x12 },
	{ 218, 0x1c, 0x1c, 0x1c }, { 219, 0x26, 0x26, 0x26 },
	{ 220, 0x30, 0x30, 0x30 }, { 221, 0x3a, 0x3a, 0x3a },
	{ 222, 0x44, 0x44, 0x44 }, { 223, 0x4e, 0x4e, 0x4e },
	{ 224, 0x58, 0x58, 0x58 }, {  36, 0x5f, 0x00, 0x00 },
	{  37, 0x5f, 0x00, 0x5f }, {  38, 0x5f, 0x00, 0x87 },
	{  39, 0x5f, 0x00, 0xaf }, {  40, 0x5f, 0x00, 0xd7 },
	{  41, 0x5f, 0x00, 0xff }, {  42, 0x5f, 0x5f, 0x00 },
	{  43, 0x5f, 0x5f, 0x5f }, {  44, 0x5f, 0x5f, 0x87 },
	{  45, 0x5f, 0x5f, 0xaf }, {  46, 0x5f, 0x5f, 0xd7 },
	{  47, 0x5f, 0x5f, 0xff }, {  48, 0x5f, 0x87, 0x00 },
	{  49, 0x5f, 0x87, 0x5f }, {  50, 0x5f, 0x87, 0x87 },
	{  51, 0x5f, 0x87, 0xaf }, {  52, 0x5f, 0x87, 0xd7 },
	{  53, 0x5f, 0x87, 0xff }, {  54, 0x5f, 0xaf, 0x00 },
	{  55, 0x5f, 0xaf, 0x5f }, {  56, 0x5f, 0xaf, 0x87 },
	{  57, 0x5f, 0xaf, 0xaf }, {  58, 0x5f, 0xaf, 0xd7 },
	{  59, 0x5f, 0xaf, 0xff }, {  60, 0x5f, 0xd7, 0x00 },
	{  61, 0x5f, 0xd7, 0x5f }, {  62, 0x5f, 0xd7, 0x87 },
	{  63, 0x5f, 0xd7, 0xaf }, {  64, 0x5f, 0xd7, 0xd7 },
	{  65, 0x5f, 0xd7, 0xff }, {  66, 0x5f, 0xff, 0x00 },
	{  67, 0x5f, 0xff, 0x5f }, {  68, 0x5f, 0xff, 0x87 },
	{  69, 0x5f, 0xff, 0xaf }, {  70, 0x5f, 0xff, 0xd7 },
	{  71, 0x5f, 0xff, 0xff }, { 225, 0x62, 0x62, 0x62 },
	{ 226, 0x6c, 0x6c, 0x6c }, { 227, 0x76, 0x76, 0x76 },
	{ 228, 0x80, 0x80, 0x80 }, {  72, 0x87, 0x00, 0x00 },
	{  73, 0x87, 0x00, 0x5f }, {  74, 0x87, 0x00, 0x87 },
	{  75, 0x87, 0x00, 0xaf }, {  76, 0x87, 0x00, 0xd7 },
	{  77, 0x87, 0x00, 0xff }, {  78, 0x87, 0x5f, 0x00 },
	{  79, 0x87, 0x5f, 0x5f }, {  80, 0x87, 0x5f, 0x87 },
	{  81, 0x87, 0x5f, 0xaf }, {  82, 0x87, 0x5f, 0xd7 },
	{  83, 0x87, 0x5f, 0xff }, {  84, 0x87, 0x87, 0x00 },
	{  85, 0x87, 0x87, 0x5f }, {  86, 0x87, 0x87, 0x87 },
	{  87, 0x87, 0x87, 0xaf }, {  88, 0x87, 0x87, 0xd7 },
	{  89, 0x87, 0x87, 0xff }, {  90, 0x87, 0xaf, 0x00 },
	{  91, 0x87, 0xaf, 0x5f }, {  92, 0x87, 0xaf, 0x87 },
	{  93, 0x87, 0xaf, 0xaf }, {  94, 0x87, 0xaf, 0xd7 },
	{  95, 0x87, 0xaf, 0xff }, {  96, 0x87, 0xd7, 0x00 },
	{  97, 0x87, 0xd7, 0x5f }, {  98, 0x87, 0xd7, 0x87 },
	{  99, 0x87, 0xd7, 0xaf }, { 100, 0x87, 0xd7, 0xd7 },
	{ 101, 0x87, 0xd7, 0xff }, { 102, 0x87, 0xff, 0x00 },
	{ 103, 0x87, 0xff, 0x5f }, { 104, 0x87, 0xff, 0x87 },
	{ 105, 0x87, 0xff, 0xaf }, { 106, 0x87, 0xff, 0xd7 },
	{ 107, 0x87, 0xff, 0xff }, { 229, 0x8a, 0x8a, 0x8a },
	{ 230, 0x94, 0x94, 0x94 }, { 231, 0x9e, 0x9e, 0x9e },
	{ 232, 0xa8, 0xa8, 0xa8 }, { 108, 0xaf, 0x00, 0x00 },
	{ 109, 0xaf, 0x00, 0x5f }, { 110, 0xaf, 0x00, 0x87 },
	{ 111, 0xaf, 0x00, 0xaf }, { 112, 0xaf, 0x00, 0xd7 },
	{ 113, 0xaf, 0x00, 0xff }, { 114, 0xaf, 0x5f, 0x00 },
	{ 115, 0xaf, 0x5f, 0x5f }, { 116, 0xaf, 0x5f, 0x87 },
	{ 117, 0xaf, 0x5f, 0xaf }, { 118, 0xaf, 0x5f, 0xd7 },
	{ 119, 0xaf, 0x5f, 0xff }, { 120, 0xaf, 0x87, 0x00 },
	{ 121, 0xaf, 0x87, 0x5f }, { 122, 0xaf, 0x87, 0x87 },
	{ 123, 0xaf, 0x87, 0xaf }, { 124, 0xaf, 0x87, 0xd7 },
	{ 125, 0xaf, 0x87, 0xff }, { 126, 0xaf, 0xaf, 0x00 },
	{ 127, 0xaf, 0xaf, 0x5f }, { 128, 0xaf, 0xaf, 0x87 },
	{ 129, 0xaf, 0xaf, 0xaf }, { 130, 0xaf, 0xaf, 0xd7 },
	{ 131, 0xaf, 0xaf, 0xff }, { 132, 0xaf, 0xd7, 0x00 },
	{ 133, 0xaf, 0xd7, 0x5f }, { 134, 0xaf, 0xd7, 0x87 },
	{ 135, 0xaf, 0xd7, 0xaf }, { 136, 0xaf, 0xd7, 0xd7 },
	{ 137, 0xaf, 0xd7, 0xff }, { 138, 0xaf, 0xff, 0x00 },
	{ 139, 0xaf, 0xff, 0x5f }, { 140, 0xaf, 0xff, 0x87 },
	{ 141, 0xaf, 0xff, 0xaf }, { 142, 0xaf, 0xff, 0xd7 },
	{ 143, 0xaf, 0xff, 0xff }, { 233, 0xb2, 0xb2, 0xb2 },
	{ 234, 0xbc, 0xbc, 0xbc }, { 235, 0xc6, 0xc6, 0xc6 },
	{ 236, 0xd0, 0xd0, 0xd0 }, { 144, 0xd7, 0x00, 0x00 },
	{ 145, 0xd7, 0x00, 0x5f }, { 146, 0xd7, 0x00, 0x87 },
	{ 147, 0xd7, 0x00, 0xaf }, { 148, 0xd7, 0x00, 0xd7 },
	{ 149, 0xd7, 0x00, 0xff }, { 150, 0xd7, 0x5f, 0x00 },
	{ 151, 0xd7, 0x5f, 0x5f }, { 152, 0xd7, 0x5f, 0x87 },
	{ 153, 0xd7, 0x5f, 0xaf }, { 154, 0xd7, 0x5f, 0xd7 },
	{ 155, 0xd7, 0x5f, 0xff }, { 156, 0xd7, 0x87, 0x00 },
	{ 157, 0xd7, 0x87, 0x5f }, { 158, 0xd7, 0x87, 0x87 },
	{ 159, 0xd7, 0x87, 0xaf }, { 160, 0xd7, 0x87, 0xd7 },
	{ 161, 0xd7, 0x87, 0xff }, { 162, 0xd7, 0xaf, 0x00 },
	{ 163, 0xd7, 0xaf, 0x5f }, { 164, 0xd7, 0xaf, 0x87 },
	{ 165, 0xd7, 0xaf, 0xaf }, { 166, 0xd7, 0xaf, 0xd7 },
	{ 167, 0xd7, 0xaf, 0xff }, { 168, 0xd7, 0xd7, 0x00 },
	{ 169, 0xd7, 0xd7, 0x5f }, { 170, 0xd7, 0xd7, 0x87 },
	{ 171, 0xd7, 0xd7, 0xaf }, { 172, 0xd7, 0xd7, 0xd7 },
	{ 173, 0xd7, 0xd7, 0xff }, { 174, 0xd7, 0xff, 0x00 },
	{ 175, 0xd7, 0xff, 0x5f }, { 176, 0xd7, 0xff, 0x87 },
	{ 177, 0xd7, 0xff, 0xaf }, { 178, 0xd7, 0xff, 0xd7 },
	{ 179, 0xd7, 0xff, 0xff }, { 237, 0xda, 0xda, 0xda },
	{ 238, 0xe4, 0xe4, 0xe4 }, { 239, 0xee, 0xee, 0xee },
	{ 180, 0xff, 0x00, 0x00 }, { 181, 0xff, 0x00, 0x5f },
	{ 182, 0xff, 0x00, 0x87 }, { 183, 0xff, 0x00, 0xaf },
	{ 184, 0xff, 0x00, 0xd7 }, { 185, 0xff, 0x00, 0xff },
	{ 186, 0xff, 0x5f, 0x00 }, { 187, 0xff, 0x5f, 0x5f },
	{ 188, 0xff, 0x5f, 0x87 }, { 189, 0xff, 0x5f, 0xaf },
	{ 190, 0xff, 0x5f, 0xd7 }, { 191, 0xff, 0x5f, 0xff },
	{ 192, 0xff, 0x87, 0x00 }, { 193, 0xff, 0x87, 0x5f },
	{ 194, 0xff, 0x87, 0x87 }, { 195, 0xff, 0x87, 0xaf },
	{ 196, 0xff, 0x87, 0xd7 }, { 197, 0xff, 0x87, 0xff },
	{ 198, 0xff, 0xaf, 0x00 }, { 199, 0xff, 0xaf, 0x5f },
	{ 200, 0xff, 0xaf, 0x87 }, { 201, 0xff, 0xaf, 0xaf },
	{ 202, 0xff, 0xaf, 0xd7 }, { 203, 0xff, 0xaf, 0xff },
	{ 204, 0xff, 0xd7, 0x00 }, { 205, 0xff, 0xd7, 0x5f },
	{ 206, 0xff, 0xd7, 0x87 }, { 207, 0xff, 0xd7, 0xaf },
	{ 208, 0xff, 0xd7, 0xd7 }, { 209, 0xff, 0xd7, 0xff },
	{ 210, 0xff, 0xff, 0x00 }, { 211, 0xff, 0xff, 0x5f },
	{ 212, 0xff, 0xff, 0x87 }, { 213, 0xff, 0xff, 0xaf },
	{ 214, 0xff, 0xff, 0xd7 }, { 215, 0xff, 0xff, 0xff },
};

int	colour_cmp_rgb(const void *, const void *);

/* Compare function for bsearch(). */
int
colour_cmp_rgb(const void *lhs0, const void *rhs0)
{
	const struct colour_rgb *lhs = lhs0, *rhs = rhs0;

	if (lhs->r < rhs->r)
		return (-1);
	if (lhs->r > rhs->r)
		return (1);

	if (lhs->g < rhs->g)
		return (-1);
	if (lhs->g > rhs->g)
		return (1);

	if (lhs->b < rhs->b)
		return (-1);
	if (lhs->b > rhs->b)
		return (1);

	return (0);
}

/* Work out the nearest colour from the 256 colour set. */
int
colour_find_rgb(u_char r, u_char g, u_char b)
{
	struct colour_rgb	rgb = { .r = r, .g = g, .b = b }, *found;
	u_int			distance, lowest, colour, i;
	int			dr, dg, db;

	found = bsearch(&rgb, colour_to_256, nitems(colour_to_256),
	    sizeof colour_to_256[0], colour_cmp_rgb);
	if (found != NULL)
		return (16 + found->i);

	colour = 16;
	lowest = UINT_MAX;
	for (i = 0; i < 240; i++) {
		dr = (int)colour_from_256[i].r - r;
		dg = (int)colour_from_256[i].g - g;
		db = (int)colour_from_256[i].b - b;

		distance = dr * dr + dg * dg + db * db;
		if (distance < lowest) {
			lowest = distance;
			colour = 16 + i;
		}
	}
	return (colour);
}

/* Set grid cell foreground colour. */
void
colour_set_fg(struct grid_cell *gc, int c)
{
	if (c & 0x100)
		gc->flags |= GRID_FLAG_FG256;
	gc->fg = c;
}

/* Set grid cell background colour. */
void
colour_set_bg(struct grid_cell *gc, int c)
{
	if (c & 0x100)
		gc->flags |= GRID_FLAG_BG256;
	gc->bg = c;
}

/* Convert colour to a string. */
const char *
colour_tostring(int c)
{
	static char	s[32];

	if (c & 0x100) {
		xsnprintf(s, sizeof s, "colour%d", c & ~0x100);
		return (s);
	}

	switch (c) {
	case 0:
		return ("black");
	case 1:
		return ("red");
	case 2:
		return ("green");
	case 3:
		return ("yellow");
	case 4:
		return ("blue");
	case 5:
		return ("magenta");
	case 6:
		return ("cyan");
	case 7:
		return ("white");
	case 8:
		return ("default");
	case 90:
		return ("brightblack");
	case 91:
		return ("brightred");
	case 92:
		return ("brightgreen");
	case 93:
		return ("brightyellow");
	case 94:
		return ("brightblue");
	case 95:
		return ("brightmagenta");
	case 96:
		return ("brightcyan");
	case 97:
		return ("brightwhite");
	}
	return (NULL);
}

/* Convert colour from string. */
int
colour_fromstring(const char *s)
{
	const char	*errstr;
	const char	*cp;
	int		 n;
	u_char		 r, g, b;

	if (*s == '#' && strlen(s) == 7) {
		for (cp = s + 1; isxdigit((u_char) *cp); cp++)
			;
		if (*cp != '\0')
			return (-1);
		n = sscanf(s + 1, "%2hhx%2hhx%2hhx", &r, &g, &b);
		if (n != 3)
			return (-1);
		return (colour_find_rgb(r, g, b) | 0x100);
	}

	if (strncasecmp(s, "colour", (sizeof "colour") - 1) == 0) {
		n = strtonum(s + (sizeof "colour") - 1, 0, 255, &errstr);
		if (errstr != NULL)
			return (-1);
		return (n | 0x100);
	}

	if (strcasecmp(s, "black") == 0 || strcmp(s, "0") == 0)
		return (0);
	if (strcasecmp(s, "red") == 0 || strcmp(s, "1") == 0)
		return (1);
	if (strcasecmp(s, "green") == 0 || strcmp(s, "2") == 0)
		return (2);
	if (strcasecmp(s, "yellow") == 0 || strcmp(s, "3") == 0)
		return (3);
	if (strcasecmp(s, "blue") == 0 || strcmp(s, "4") == 0)
		return (4);
	if (strcasecmp(s, "magenta") == 0 || strcmp(s, "5") == 0)
		return (5);
	if (strcasecmp(s, "cyan") == 0 || strcmp(s, "6") == 0)
		return (6);
	if (strcasecmp(s, "white") == 0 || strcmp(s, "7") == 0)
		return (7);
	if (strcasecmp(s, "default") == 0 || strcmp(s, "8") == 0)
		return (8);
	if (strcasecmp(s, "brightblack") == 0 || strcmp(s, "90") == 0)
		return (90);
	if (strcasecmp(s, "brightred") == 0 || strcmp(s, "91") == 0)
		return (91);
	if (strcasecmp(s, "brightgreen") == 0 || strcmp(s, "92") == 0)
		return (92);
	if (strcasecmp(s, "brightyellow") == 0 || strcmp(s, "93") == 0)
		return (93);
	if (strcasecmp(s, "brightblue") == 0 || strcmp(s, "94") == 0)
		return (94);
	if (strcasecmp(s, "brightmagenta") == 0 || strcmp(s, "95") == 0)
		return (95);
	if (strcasecmp(s, "brightcyan") == 0 || strcmp(s, "96") == 0)
		return (96);
	if (strcasecmp(s, "brightwhite") == 0 || strcmp(s, "97") == 0)
		return (97);
	return (-1);
}

/* Convert 256 colour palette to 16. */
u_char
colour_256to16(u_char c)
{
	static const u_char table[256] = {
		 0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
		 0,  4,  4,  4, 12, 12,  2,  6,  4,  4, 12, 12,  2,  2,  6,  4,
		12, 12,  2,  2,  2,  6, 12, 12, 10, 10, 10, 10, 14, 12, 10, 10,
		10, 10, 10, 14,  1,  5,  4,  4, 12, 12,  3,  8,  4,  4, 12, 12,
		 2,  2,  6,  4, 12, 12,  2,  2,  2,  6, 12, 12, 10, 10, 10, 10,
		14, 12, 10, 10, 10, 10, 10, 14,  1,  1,  5,  4, 12, 12,  1,  1,
		 5,  4, 12, 12,  3,  3,  8,  4, 12, 12,  2,  2,  2,  6, 12, 12,
		10, 10, 10, 10, 14, 12, 10, 10, 10, 10, 10, 14,  1,  1,  1,  5,
		12, 12,  1,  1,  1,  5, 12, 12,  1,  1,  1,  5, 12, 12,  3,  3,
		 3,  7, 12, 12, 10, 10, 10, 10, 14, 12, 10, 10, 10, 10, 10, 14,
		 9,  9,  9,  9, 13, 12,  9,  9,  9,  9, 13, 12,  9,  9,  9,  9,
		13, 12,  9,  9,  9,  9, 13, 12, 11, 11, 11, 11,  7, 12, 10, 10,
		10, 10, 10, 14,  9,  9,  9,  9,  9, 13,  9,  9,  9,  9,  9, 13,
		 9,  9,  9,  9,  9, 13,  9,  9,  9,  9,  9, 13,  9,  9,  9,  9,
		 9, 13, 11, 11, 11, 11, 11, 15,  0,  0,  0,  0,  0,  0,  8,  8,
		 8,  8,  8,  8,  7,  7,  7,  7,  7,  7, 15, 15, 15, 15, 15, 15
	};

	return (table[c]);
}
